package com.moloco.sdk.internal.services.init

import android.net.Uri
import com.moloco.sdk.Init
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.internal.Error
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.appendXMolocoUserAgent
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.AppInfoService
import com.moloco.sdk.internal.services.DeviceInfoService
import com.moloco.sdk.internal.services.usertracker.UserTrackerService
import com.moloco.sdk.publisher.MediationInfo
import com.moloco.sdk.xenoss.sdkdevkit.android.core.requestTimeoutMillis
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.HttpRequestTimeoutException
import io.ktor.client.request.get
import io.ktor.client.request.headers
import io.ktor.http.HttpStatusCode
import kotlinx.coroutines.withContext

internal fun interface InitApi {
    companion object {
        // 3s as our P99 is 1.5s (so 2x our P99)
        internal const val httpRequestTimeout = 3000L
    }
    /**
     * @param mediationInfo [MediationInfo][com.moloco.sdk.publisher.MediationInfo] required for Moloco SDK, when running in not-direct-integration/bid-token mode.
     */
    suspend operator fun invoke(
        appKey: String,
        mediationInfo: MediationInfo?
    ): Result<Init.SDKInitResponse, Error>
}

private const val TAG = "InitApi"
internal class InitApiImpl(
    private val deviceInfoService: DeviceInfoService,
    private val appInfoService: AppInfoService,
    private val userTrackerService: UserTrackerService,
    private val sdkVersion: String,
    endpoint: String,
    private val requestTimeoutMillis: Long,
    private val httpClient: HttpClient,
) : InitApi {

    private val endpointUri = Uri.parse(endpoint)

    override suspend fun invoke(
        appKey: String,
        mediationInfo: MediationInfo?
    ): Result<Init.SDKInitResponse, Error>{
        val httpRequestTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.SDKInitHttpRequest.eventName)
        return try {
            val deviceInfo = deviceInfoService()
            val appInfo = appInfoService()
            val mref = userTrackerService.getIdentifier()
            MolocoLogger.debug(
                TAG,
                "Requesting Init with appKey: $appKey, mref: $mref, url: $endpointUri"
            )

            val response = httpClient.get(
                endpointUri.buildUpon()
                    .appendQueryParameter("app_key", appKey)
                    .appendQueryParameter("rid", mref)
                    .build()
                    .toString()
            ) {
                headers {
                    appendXMolocoUserAgent(
                        sdkVersion,
                        deviceInfo.osVersion,
                        mediationInfo = mediationInfo
                    )
                    append("X-Moloco-App-Bundle", appInfo.packageName)
                }
                requestTimeoutMillis(requestTimeoutMillis)
            }

            when (val httpStatus = response.status) {
                HttpStatusCode.OK -> {
                    AndroidClientMetrics.recordTimerEvent(httpRequestTimerEvent.withTag(AcmTag.Result.tagName, "success"))
                    Result.Success(
                        withContext(DispatcherProvider().io) {
                            MolocoLogger.info(TAG, "Successful Init")
                            // Switch to ByteReadChannel if going out of memory.
                            Init.SDKInitResponse.parseFrom(response.body<ByteArray>())
                        }
                    )
                }

                HttpStatusCode.NotFound -> {
                    AndroidClientMetrics.recordTimerEvent(httpRequestTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "${httpStatus.value}"))
                    val errorDescription =
                        "http status $httpStatus: App not found or AppKey is not correct"
                    MolocoLogger.error(
                        TAG,
                        errorDescription
                    )
                    Result.Failure(Error(errorDescription, httpStatus.value))
                }

                else -> {
                    AndroidClientMetrics.recordTimerEvent(httpRequestTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "${httpStatus.value}"))
                    val errorDescription = "http status: $httpStatus"
                    MolocoLogger.error(TAG, errorDescription)
                    Result.Failure(Error(errorDescription, httpStatus.value))
                }
            }
        } catch (requestTimeoutException: HttpRequestTimeoutException) {
            val httpStatusCode = HttpStatusCode.RequestTimeout.value
            AndroidClientMetrics.recordTimerEvent(httpRequestTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "$httpStatusCode"))
            MolocoLogger.error(TAG, requestTimeoutException.message.toString(), requestTimeoutException)
            Result.Failure(Error(requestTimeoutException.toString(), httpStatusCode))
        } catch (e: Exception) {
            val httpStatusCode = HttpStatusCode.BadRequest.value
            AndroidClientMetrics.recordTimerEvent(httpRequestTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "$httpStatusCode"))
            MolocoLogger.error(TAG, e.message.toString(), e)
            Result.Failure(Error(e.toString(), httpStatusCode))
        }
    }

}
