package com.moloco.sdk.internal.services.init

import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.Error
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.publisher.MediationInfo
import io.ktor.http.HttpStatusCode
import kotlinx.coroutines.delay

/**
 * Stateful service that provides capability to invoke init API the SDK as well as cache
 * the last successful init response
 */
interface InitService {
    /**
     * @return null if the SDK hasn't been initialized. non-null if SDK has successfully init
     */
    val lastInitResponse: SDKInitResponse?

    /**
     * @param mediationInfo [MediationInfo][com.moloco.sdk.publisher.MediationInfo] required for Moloco SDK, when running in not-direct-integration/bid-token mode.
     * @return Performs init with backend and returns the init response
     */
    suspend fun performInit(
        appKey: String,
        mediationInfo: MediationInfo?
    ): Result<SDKInitResponse, String>
}

private const val TAG = "InitService"
internal class InitServiceImpl(val initApi: InitApi) : InitService {
    private var _lastInitResponse: SDKInitResponse? = null
    override val lastInitResponse: SDKInitResponse?
        get() = _lastInitResponse

    override suspend fun performInit(
        appKey: String,
        mediationInfo: MediationInfo?
    ): Result<SDKInitResponse, String> {
        lateinit var result: Result<SDKInitResponse, Error>

        // Try initializing up to 3 times
        // https://docs.google.com/document/d/1EqQfiptDtcgjwWNET4FuYjQSqUXsxcCbT1JJY67veqA/edit#heading=h.iidd1pqa86oj
        repeat(3) { attempt ->
            result = initApi(appKey, mediationInfo = mediationInfo)

            when (result) {
                is Result.Success -> // If initialization succeeds, return early from the coroutine
                    (result as Result.Success<SDKInitResponse, Error>).value.let {
                        MolocoLogger.info(TAG, "Init, successful in attempt(#$attempt)")
                        _lastInitResponse = it
                        AndroidClientMetrics.recordCountEvent(
                            CountEvent(AcmCount.SDKPerformInitAttempt.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.RetryAttempt.tagName, "$attempt")
                        )
                        return@performInit Result.Success(it)
                    }
                is Result.Failure -> (result as Result.Failure<SDKInitResponse, Error>).let {
                    val failureCode = it.value.failureCode
                    AndroidClientMetrics.recordCountEvent(
                        CountEvent(AcmCount.SDKPerformInitAttempt.eventName)
                            .withTag(AcmTag.Result.tagName, "failure")
                            .withTag(AcmTag.RetryAttempt.tagName, "$attempt")
                            .withTag(AcmTag.Reason.tagName, "$failureCode")
                    )
                    MolocoLogger.info(
                        TAG,
                        "Init attempt(#$attempt) failed with error: $failureCode"
                    )

                    // If the error is not retryable, short-circuit and return terminal failure
                    if (!failureCode.isRetryable()) {
                        MolocoLogger.info(
                            TAG,
                            "Init response is non-retryable failure: $failureCode"
                        )
                        return Result.Failure((result as Result.Failure).value.description)
                    }
                }
            }

            // Otherwise, wait for 1 second before retrying
            delay(1000)
        }

        val terminalFailure: Result.Failure<SDKInitResponse, String> = Result.Failure(
            (result as Result.Failure<SDKInitResponse, Error>).value.description
        )
        MolocoLogger.info(TAG, "Moloco SDK Init failed after all retries: ${terminalFailure.value}")
        return terminalFailure
    }
}

internal fun Int.isRetryable(): Boolean = this == HttpStatusCode.TooManyRequests.value ||
        this == HttpStatusCode.RequestTimeout.value || this < 400 || this >= 500
