package com.moloco.sdk.internal.services.init

import android.net.Uri
import com.moloco.sdk.Init.SDKInitResponse.Region
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.HttpRequestClient

internal interface InitTrackingApi {
    /**
     * @param errorCode ENUM or Code for reason for failed initialization.
     * @param region region of notifnt server (US, EU, Asia) based on initialization country. "US" by default if region info unavailable.
     */
    operator fun invoke(initStatus: Result<Unit, Int?>, region: Region = DefaultRegion)

    companion object {
        val DefaultRegion = Region.US
    }
}

private const val TAG = "InitTrackingApi"
internal class InitTrackingApiImpl(
    private val endpoint: String,
    private val applicationPackageName: String,
    private val httpRequestClient: HttpRequestClient,
) : InitTrackingApi {

    override fun invoke(initStatus: Result<Unit, Int?>, region: Region) {
        try {
            logInitStatus(initStatus)

            val regionString = region.stringify() ?: return

            val preparedUrl = Uri.parse(endpoint.replace("{{region}}", regionString))
                .buildUpon()
                .appendQueryParameter("package_name", applicationPackageName)
                .appendQueryParameter(
                    "status",
                    if (initStatus is Result.Success) "true" else "false"
                )
                .run {
                    if (initStatus is Result.Failure && initStatus.value != null) {
                        appendQueryParameter("err_code", initStatus.value.toString())
                    } else {
                        this
                    }
                }
                .build()

            httpRequestClient.send(preparedUrl.toString())
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "invoke()", e)
        }
    }

    companion object {
        private fun Region.stringify(): String? = when (this) {
            Region.US -> "us"
            Region.EU -> "eu"
            Region.ASIA -> Asia
            Region.INDIA -> Asia
            else -> null
        }

        private const val Asia = "asia"
    }

    private fun logInitStatus(initStatus: Result<Unit, Int?>) {
        when (initStatus) {
            is Result.Success -> {
                MolocoLogger.debug(TAG, "Reporting InitTracking success")
            }
            is Result.Failure -> {
                MolocoLogger.debug(TAG, "Reporting InitTracking failure: ${initStatus.value}")
            }
        }
    }
}
