package com.moloco.sdk.publisher

import android.app.Activity
import android.content.Context
import android.view.View
import android.widget.FrameLayout
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow

/**
 * A banner ad implementation which extends the Android [FrameLayout] and implements [AdLoad]
 * and [Destroyable] interfaces.
 *
 * General usage of banners:
 * 1. [Moloco.createBanner()][Moloco.createBanner] instance first
 * 2. add banner to the view
 * 3. attach an optional [adShowListener]
 * 4. Fetch ortb bid response
 * 5. [load()][AdLoad.load] bid response. [AdLoad.Listener] or [isLoaded][AdLoad.isLoaded] can be used for load status check
 * 6. Once banner is not needed anymore [destroy()][Destroyable.destroy] it to prevent memory leaks
 *
 * Things to note:
 *
 * - Once banner is loaded it'll automatically display an ad as long as its added to the screen and visible
 *
 * - In order to load an another ad, call [load()][AdLoad.load] again provided with a new bid response
 *
 * - [load()][AdLoad.load] calls, their amount and order are independent of banner's adding/removal to the view hierarchy;
 * in other words, any order of the mentioned operations is allowed.
 *
 * @param activity The activity to which the banner is attached
 *
 * @sample com.moloco.sdk.publisher.BannerActivitySample
 */
abstract class Banner(context: Context) : FrameLayout(context), AdLoad, Destroyable {
    init {
        tag = "MolocoBannerView"
    }

    /**
     * The listener to handle banner ad show/hide events
     */
    open var adShowListener: BannerAdShowListener? = null

    private val _isViewShown = MutableStateFlow(false)

    /**
     *  Provides [View.isShown][android.view.View.isShown] latest value and follow-up updates
     */
    open val isViewShown = _isViewShown.asStateFlow()

    override fun onVisibilityChanged(changedView: View, visibility: Int) {
        _isViewShown.value = changedView.isShown
    }
}
