package com.moloco.sdk.publisher

/**
 * Interface for full-screen ads that can be shown to users.
 *
 * @param T the listener that will be notified of the ad show events.
 */
interface FullscreenAd<in T : AdShowListener> : AdLoad, Destroyable {

    /**
     * Shows the ad to the user.
     *
     * Note: If the underlying ad experiences an error during the displaying stage,
     * the underlying ad object will be destroyed and the [AdShowListener.onAdShowFailed]
     * and [AdShowListener.onAdHidden] callbacks will be fired.
     *
     * @param listener the listener that will be notified of the ad show events.
     */
    fun show(listener: T?)
}

/**
 * Interface for interstitial ads, a type of full-screen ad.
 *
 * General usage of insterstitials:
 * 1. [Moloco.createInterstitial()][Moloco.createInterstitial] instance first
 * 2. Fetch ortb bid response
 * 3. [load()][AdLoad.load] bid response. [AdLoad.Listener] or [isLoaded][AdLoad.isLoaded] can be used for load status check
 * 4. [show()][FullscreenAd.show] ad when needed. You can also provide an optional [InterstitialAdShowListener].
 * 5. Once ad is not needed anymore [destroy()][Destroyable.destroy] it to prevent memory leaks
 *
 * Things to note:
 *
 * - In order to load an another ad, call [load()][AdLoad.load] again provided with a new bid response
 *
 * - [load()][AdLoad.load] call will force finish and hide the currently displaying ad.
 * Therefore, if there's a need to implement preloading of multiple ads,
 * consider creation of an additional interstitial ad instance via [Moloco.createInterstitial()][Moloco.createInterstitial]
 *
 * @sample com.moloco.sdk.publisher.InterstitialAdActivitySample
 */
interface InterstitialAd : FullscreenAd<InterstitialAdShowListener>

/**
 * Interface for rewarded interstitial ads, a type of full-screen ad that provides a reward to the user
 * after they interact with it.
 *
 * General usage of rewarded insterstitials:
 * 1. [Moloco.createRewardedInterstitial] instance first
 * 2. Fetch ortb bid response
 * 3. [load()][AdLoad.load] bid response. [AdLoad.Listener] or [isLoaded][AdLoad.isLoaded] can be used for load status check
 * 4. [show()][FullscreenAd.show] ad when needed. You can also provide an optional [InterstitialAdShowListener].
 * 5. Once ad is not needed anymore [destroy()][Destroyable.destroy] it to prevent memory leaks
 *
 * Things to note:
 *
 * - In order to load an another ad, call [load()][AdLoad.load] again provided with a new bid response
 *
 * - [load()][AdLoad.load] call will force finish and hide the currently displaying ad.
 * Therefore, if there's a need to implement preloading of multiple ads,
 * consider creation of an additional rewarded interstitial ad instance via [Moloco.createRewardedInterstitial()][Moloco.createRewardedInterstitial]
 *
 * @sample com.moloco.sdk.publisher.RewardedInterstitialAdSample
 */
interface RewardedInterstitialAd : FullscreenAd<RewardedInterstitialAdShowListener>
