package com.moloco.sdk.publisher

import android.view.View

/**
 * Interface representing a native ad for mediation.
 *
 * Implementations of this interface should provide information about the native ad's assets such as
 * title, description, sponsor text, call to action text, rating, icon URI, main image URI, and video view.
 * These properties should return null in case the ad is not loaded, the asset is not available in the bid response,
 * or not preloaded successfully.
 */
interface NativeAdForMediation : AdLoad, NativeAdOrtbRequestRequirements, Destroyable {

    /**
     * Title for the ad
     */
    val title: String?

    /**
     * Text for the ad's body
     */
    val description: String?

    /**
     * Advertiser for the ad
     */
    val sponsorText: String?

    /**
     * Call to action text
     */
    val callToActionText: String?

    /**
     * Rating for the ad
     */
    val rating: Float?

    /**
     * Uri link to the icon of the ad
     */
    val iconUri: String?

    /**
     * For medium sized native ads the image to be used if [video] is null
     */
    val mainImageUri: String?

    /**
     * For medium sized native ads if [video] is not null then this should be used instead of [mainImageUri]
     */
    val video: View?

    /**
     * Call this when adapter's view (or some of the asset views, it's up to you to decide) gets rendered.
     */
    fun handleImpression()

    /**
     * Call this when adapter's view implementation get clicked.
     */
    fun handleGeneralAdClick()

    /**
     * Call this when icon image asset view gets clicked.
     */
    fun handleIconClick()

    /**
     * Call this when main image asset view gets clicked.
     */
    fun handleMainImageClick()

    /**
     * Register this listener, to be informed of [onImpressionHandled] or [onGeneralClickHandled] events
     */
    var interactionListener: InteractionListener?

    /**
     * Listener to be informed of [onImpressionHandled] or [onGeneralClickHandled] events
     */
    interface InteractionListener {
        /**
         * Get's called if the latest [handleImpression] call registered impression.
         * You can use this for counting impression on adapter's side.
         *
         */
        fun onImpressionHandled()

        /**
         * Get's called if the latest [handleGeneralAdClick] call registered click.
         * You can use this for counting clicks on adapter's side.
         *
         */
        fun onGeneralClickHandled()
    }
}

// TODO.
//  - Common code extraction for this, Banner, Fullscreen ad;
//  - current implementations of ads have load and rendering parts fused together,
//  as you can see, we need to support independent load and render actions
//  (respective code should be independent as well), as opposed to what we have now.
