package com.moloco.sdk.publisher

/**
 * Provides information for a proper generation of the ortb native ad request: asset types, their amount, requred or not, ids etc.
 * Main goal is to provide a proper mapping between bid request and bid response native assets, tracking events etc.
 */
interface NativeAdOrtbRequestRequirements {

    /**
     * Native ad ortb request requirements; use this property to generate a proper ortb bid request for native ads
     * in order to get a compatible bid response for [`load(bidResponse)`][AdLoad.load] call
     * according to [_OpenRTB Dynamic Native Ads API Specification_](https://www.iab.com/wp-content/uploads/2018/03/OpenRTB-Native-Ads-Specification-Final-1.2.pdf)
     */
    val nativeAdOrtbRequestRequirements: Requirements

    /**
     * Respective classes and field names required for a proper ortb bid request generation according to
     * [_OpenRTB Dynamic Native Ads API Specification_](https://www.iab.com/wp-content/uploads/2018/03/OpenRTB-Native-Ads-Specification-Final-1.2.pdf)
     * @param assets - map of assets required for a proper ad rendering; key - [Asset.id]
     */
    class Requirements(
        val assets: Map<Int, Asset>,
        val eventTrackers: List<EventTracker>
    ) {

        sealed class Asset(
            val id: Int,
            val required: Boolean,
        ) {
            class Title(id: Int, required: Boolean, val length: Int) : Asset(id, required)

            class Image(
                id: Int,
                required: Boolean,
                val type: Int?
            ) : Asset(id, required) {
                internal companion object {
                    // Asset image type constant for requesting video thumbnail asset.
                    const val TypeMolocoThumbnail = 501
                }
            }

            class Video(
                id: Int,
                required: Boolean
            ) : Asset(id, required)

            class Data(
                id: Int,
                required: Boolean,
                val type: Int,
                val len: Int?
            ) : Asset(id, required)
        }

        class EventTracker(val eventType: Int, val methodTypes: List<Int>)
    }
}
