package com.moloco.sdk.service_locator

import android.annotation.SuppressLint
import android.content.Context
import androidx.datastore.preferences.core.PreferenceDataStoreFactory
import androidx.datastore.preferences.preferencesDataStoreFile
import androidx.lifecycle.ProcessLifecycleOwner
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.Init
import com.moloco.sdk.internal.AdFactory
import com.moloco.sdk.internal.AdFactoryImpl
import com.moloco.sdk.internal.ViewLifecycleOwner
import com.moloco.sdk.internal.ViewLifecycleOwnerImpl
import com.moloco.sdk.internal.android_context.ApplicationContext
import com.moloco.sdk.internal.applicationPackageName
import com.moloco.sdk.internal.error.ErrorReportingService
import com.moloco.sdk.internal.error.ErrorReportingServiceImpl
import com.moloco.sdk.internal.error.api.ErrorReportingApiImpl
import com.moloco.sdk.internal.error.crash.CrashDetectorService
import com.moloco.sdk.internal.error.crash.CrashDetectorServiceImpl
import com.moloco.sdk.internal.error.crash.CrashHandlerServiceImpl
import com.moloco.sdk.internal.error.crash.filters.MolocoSDKExceptionFilter
import com.moloco.sdk.internal.http.buildHttpClient
import com.moloco.sdk.internal.services.AdDataService
import com.moloco.sdk.internal.services.AdDataServiceImpl
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTrackerImpl
import com.moloco.sdk.internal.services.AndroidDeviceInfoService
import com.moloco.sdk.internal.services.AppInfoService
import com.moloco.sdk.internal.services.AppInfoServiceImpl
import com.moloco.sdk.internal.services.AudioService
import com.moloco.sdk.internal.services.AudioServiceImpl
import com.moloco.sdk.internal.services.ConnectivityService
import com.moloco.sdk.internal.services.ConnectivityServiceImpl
import com.moloco.sdk.internal.services.DataStoreService
import com.moloco.sdk.internal.services.DeviceInfoService
import com.moloco.sdk.internal.services.NetworkInfoService
import com.moloco.sdk.internal.services.NetworkInfoServiceImpl
import com.moloco.sdk.internal.services.PreferencesDataStoreServiceImpl
import com.moloco.sdk.internal.services.ScreenInfoService
import com.moloco.sdk.internal.services.ScreenInfoServiceImpl
import com.moloco.sdk.internal.services.SingleObserverBackgroundThenForegroundAnalyticsListener
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.services.TimeProviderServiceImpl
import com.moloco.sdk.internal.services.analytics.AnalyticsService
import com.moloco.sdk.internal.services.analytics.AnalyticsServiceImpl
import com.moloco.sdk.internal.services.config.ConfigService
import com.moloco.sdk.internal.services.config.RemoteConfigService
import com.moloco.sdk.internal.services.events.CustomUserEventBuilderServiceImpl
import com.moloco.sdk.internal.services.events.CustomUserEventConfigServiceImpl
import com.moloco.sdk.internal.services.init.InitApi
import com.moloco.sdk.internal.services.init.InitApiImpl
import com.moloco.sdk.internal.services.init.InitService
import com.moloco.sdk.internal.services.init.InitServiceImpl
import com.moloco.sdk.internal.services.init.InitTrackingApi
import com.moloco.sdk.internal.services.init.InitTrackingApiImpl
import com.moloco.sdk.internal.services.proto.ProtoEncoderService
import com.moloco.sdk.internal.services.proto.ProtoEncoderServiceImpl
import com.moloco.sdk.internal.services.usertracker.UUIDUserIdentifierGenerator
import com.moloco.sdk.internal.services.usertracker.UserIdentifierRepository
import com.moloco.sdk.internal.services.usertracker.UserIdentifierRepositoryImpl
import com.moloco.sdk.internal.services.usertracker.UserTrackerIdentifierGenerator
import com.moloco.sdk.internal.services.usertracker.UserTrackerService
import com.moloco.sdk.internal.services.usertracker.UserTrackerServiceImpl
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.service_locator.SdkObjectFactory.DeviceAndApplicationInfo.appInfoSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.DeviceAndApplicationInfo.screenInfoSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.Miscellaneous.adDataSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.Miscellaneous.protoEncoderSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.Miscellaneous.timeProviderSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.Network.networkInfoSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.Network.persistentHttpRequestSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.Storage.dataStoreSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.UserTracking.customUserEventBuilderFactory
import com.moloco.sdk.service_locator.SdkObjectFactory.UserTracking.customUserEventConfigSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.UserTracking.userTrackingSingleton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.DECLoader
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandlerImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTrackerImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.ChunkedMediaDownloaderImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.DefaultMediaConfig
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.LegacyMediaDownloader
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MEDIA_CACHE_DIR
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheLocationProviderImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepositoryImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaConfig
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventConfigService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.BestAttemptPersistentHttpRequest
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentWorker
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentWorkerImpl
import io.ktor.client.HttpClient

/**
 * An in-house service locator named Jimbo, designed to replace Koin.
 */
@Suppress("MemberVisibilityCanBePrivate")
@SuppressLint("StaticFieldLeak")
internal object SdkObjectFactory {
    /**
     * Returns [Context.getApplicationContext]
     */
    val context: Context get() = ApplicationContext()

    // Init
    object Initialization {
        @Volatile
        private var adFactorySingleton: AdFactory? = null
        fun adFactorySingleton(initResponse: Init.SDKInitResponse): AdFactory {
            return adFactorySingleton ?: synchronized(this) {
                adFactorySingleton
                    ?: AdFactoryImpl(initResponse, customUserEventBuilderFactory).also {
                        adFactorySingleton = it
                    }
            }
        }

        val trackingApiSingleton: InitTrackingApi by lazy {
            InitTrackingApiImpl(
                BuildConfig.MOLOCO_ENDPOINT_INIT_TRACKING,
                applicationPackageName(),
                BestAttemptPersistentHttpRequest()
            )
        }

        val initApiSingleton: InitApi by lazy {
            InitApiImpl(
                deviceInfoService = deviceInfoSingleton,
                appInfoService = appInfoSingleton,
                userTrackerService = userTrackingSingleton,
                sdkVersion = BuildConfig.SDK_VERSION_NAME,
                endpoint = BuildConfig.MOLOCO_ENDPOINT_INIT_CONFIG,
                requestTimeoutMillis = InitApi.httpRequestTimeout,
                httpClient = Network.httpClientSingleton
            )
        }

        val initSingleton: InitService by lazy {
            InitServiceImpl(initApiSingleton)
        }
    }

    // Ad Load related objects
    object AdLoadModule {
        val decLoader
        get() = DECLoader(
            mediaCacheRepository = Media.mediaCacheRepository,
            errorReportingService = Analytics.errorReportingSingleton
        )
    }

    // Device and app info
    object DeviceAndApplicationInfo {
        val appInfoSingleton: AppInfoService by lazy {
            AppInfoServiceImpl(context)
        }
        val deviceInfoSingleton: DeviceInfoService by lazy {
            AndroidDeviceInfoService(context)
        }
        val audioSingleton: AudioService by lazy {
            AudioServiceImpl(context)
        }
        val screenInfoSingleton: ScreenInfoService by lazy {
            ScreenInfoServiceImpl(context)
        }
        val connectivityService: ConnectivityService by lazy {
            ConnectivityServiceImpl(context)
        }
    }

    // Analytics
    object Analytics {
        val analyticsSingleton: AnalyticsService by lazy {
            AnalyticsServiceImpl(persistentHttpRequestSingleton, customUserEventBuilderFactory, customUserEventConfigSingleton)
        }
        val singleObserverBgFgListenerSingleton by lazy {
            SingleObserverBackgroundThenForegroundAnalyticsListener(analyticsSingleton, timeProviderSingleton)
        }
        val applicationLifecycleTrackerSingleton: AnalyticsApplicationLifecycleTracker by lazy {
            AnalyticsApplicationLifecycleTrackerImpl(ProcessLifecycleOwner.get().lifecycle, singleObserverBgFgListenerSingleton)
        }

        val errorReportingSingleton: ErrorReportingService by lazy {
            ErrorReportingServiceImpl(
                Config.configService,
                ErrorReportingApiImpl(timeProviderSingleton, BestAttemptPersistentHttpRequest())
            )
        }
    }

    // User Tracking
    object UserTracking {
        val userIdentifierRepositorySingleton: UserIdentifierRepository by lazy {
            UserIdentifierRepositoryImpl(dataStoreSingleton)
        }
        val userTrackingIdentifierGeneratorFactory: UserTrackerIdentifierGenerator
            get() = UUIDUserIdentifierGenerator()
        val userTrackingSingleton: UserTrackerService by lazy {
            UserTrackerServiceImpl(userTrackingIdentifierGeneratorFactory, userIdentifierRepositorySingleton)
        }
        val customUserEventConfigSingleton: CustomUserEventConfigService by lazy {
            CustomUserEventConfigServiceImpl()
        }
        val customUserEventBuilderFactory: CustomUserEventBuilderService
            get() = CustomUserEventBuilderServiceImpl(
                appInfoService = appInfoSingleton,
                networkInfoService = networkInfoSingleton,
                deviceInfoService = deviceInfoSingleton,
                screenInfoService = screenInfoSingleton,
                userIdentifierService = userTrackingSingleton,
                adDataService = adDataSingleton,
                encoderService = protoEncoderSingleton,
                userEventConfigService = customUserEventConfigSingleton,
                sdkVersion = BuildConfig.SDK_VERSION_NAME)
    }

    // Network
    object Network {
        val httpClientSingleton: HttpClient by lazy {
            buildHttpClient(appInfoSingleton(), deviceInfoSingleton(), Moloco.appKey, Moloco.mediationInfo)
        }

        val networkInfoSingleton: NetworkInfoService by lazy {
            NetworkInfoServiceImpl(context, deviceInfoSingleton)
        }

        val persistentWorkerSingleton: PersistentWorker by lazy {
            PersistentWorkerImpl(context)
        }

        val persistentHttpRequestSingleton: PersistentHttpRequest by lazy {
            PersistentHttpRequest.create(persistentWorkerSingleton)
        }
    }

    // Data storage
    object Storage {
        val dataStoreSingleton: DataStoreService by lazy {
            val preferencesName = "moloco_sdk_preferences"

            val dataStore = PreferenceDataStoreFactory.create(
                produceFile = { context.preferencesDataStoreFile(preferencesName) }
            )
            PreferencesDataStoreServiceImpl(dataStore)
        }
    }

    // Provides services and objects related to Media handling
    object Media {
        val mediaConfigSingleton: MediaConfig by lazy {
            Config.configService.getConfig(MediaConfig::class.java, DefaultMediaConfig)
        }

        val mediaCacheRepository: MediaCacheRepository by lazy {
            val chunkedMediaDownloaderImpl = ChunkedMediaDownloaderImpl(
                mediaConfigSingleton,
                DeviceAndApplicationInfo.connectivityService,
                Analytics.errorReportingSingleton,
                Network.httpClientSingleton,
            )
            MediaCacheRepositoryImpl(
                mediaConfig = mediaConfigSingleton,
                legacyMediaDownloader = LegacyMediaDownloader(
                    DeviceAndApplicationInfo.connectivityService,
                    Analytics.errorReportingSingleton,
                    Network.httpClientSingleton,
                ),
                chunkedMediaDownloader = chunkedMediaDownloaderImpl,
                mediaCacheLocationProvider = MediaCacheLocationProviderImpl(context, MEDIA_CACHE_DIR)
            )
        }
    }

    // Provides services and objects related to Config handling
    object Config {
        val configService: ConfigService by lazy {
            RemoteConfigService()
        }
    }

    // Misc
    object Miscellaneous {
        val timeProviderSingleton: TimeProviderService by lazy {
            TimeProviderServiceImpl()
        }

        val protoEncoderSingleton: ProtoEncoderService by lazy {
            ProtoEncoderServiceImpl()
        }
        val adDataSingleton: AdDataService by lazy {
            AdDataServiceImpl(context)
        }

        val viewLifecycleOwnerSingleton:ViewLifecycleOwner by lazy {
            ViewLifecycleOwnerImpl()
        }

        val externalLinkHandlerFactory: ExternalLinkHandler
            get() = ExternalLinkHandlerImpl(context)

        val viewVisibilityTrackerFactory: ViewVisibilityTracker
            get() = ViewVisibilityTrackerImpl()
    }

    object CrashReporting {
        val crashDetectorSingleton: CrashDetectorService by lazy {
            CrashDetectorServiceImpl(
                CrashHandlerServiceImpl(listOf(MolocoSDKExceptionFilter()),
                ErrorReportingApiImpl(timeProviderSingleton, BestAttemptPersistentHttpRequest()))
            )
        }
    }
}
