package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer

import android.content.Context
import android.view.View
import android.view.ViewGroup
import android.view.ViewGroup.LayoutParams.MATCH_PARENT
import android.widget.FrameLayout
import androidx.compose.ui.platform.ComposeView
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.time.Duration

/**
 * Banner container for ad rendering. Works almost the same as [FullscreenAd] implementations <br>
 * The only difference is that instead of calling show() function, call [android.view.View.setVisibility] = true (false or detach from view hierarchy stops displaying an ad)
 */
abstract class Banner<T : AdShowListener>(
    context: Context,
) : FrameLayout(context), CreativeTypeProvider, AdLoad, AdDisplayState, Destroyable {

    open var adShowListener: T? = null

    protected val scope = CoroutineScope(DispatcherProvider().main)

    protected var adView: View? = null
        set(adView) {
            val oldAdView = field
            field = adView

            removeAllViews()
            // TODO. Just to be on the safe side. Should be called outside though
            //  and in the perfect world, we should not know anything about implementation details (ComposeView).
            // https://developer.android.com/reference/kotlin/androidx/compose/ui/platform/ComposeView:
            // Call disposeComposition to dispose of the underlying composition earlier,
            // or if the view is never initially attached to a window. (The requirement to dispose of the composition explicitly in the event that the view is never (re)attached is temporary.)
            (oldAdView as? ComposeView)?.disposeComposition()

            adView?.let {
                addView(it, ViewGroup.LayoutParams(MATCH_PARENT, MATCH_PARENT))
            }
        }

    protected abstract val adLoader: AdLoad

    override val isLoaded by lazy { adLoader.isLoaded }

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        scope.launch {
            adLoader.load(timeout, listener)
            isLoaded.first { it }
            prepareAdViewForDisplay()
        }
    }

    abstract fun prepareAdViewForDisplay()

    override fun onVisibilityChanged(changedView: View, visibility: Int) {
        super.onVisibilityChanged(changedView, visibility)
        isViewVisible.value = visibility == VISIBLE
    }

    /**
     * Denotes if the view is visible.
     * Note: If its parent is [View.INVISIBLE] or [View.GONE], this will still return `true` if its visibility is set to [View.VISIBLE]
     */
    private val isViewVisible = MutableStateFlow(false)

    /**
     * Denotes if the view is visible.
     * Note: If its parent is [View.INVISIBLE] or [View.GONE], this will still return `true` if its visibility is set to [View.VISIBLE]
     */
    override val isAdDisplaying by lazy {
        isLoaded.combine(isViewVisible) { isLoaded, isViewShown ->
            isLoaded && isViewShown
        }.stateIn(
            scope,
            SharingStarted.Eagerly,
            false
        )
    }

    /**
     * Releases all unmanaged resources and removes itself from the view hierarchy.
     */
    override fun destroy() {
        scope.cancel()
        adView = null

        (parent as? ViewGroup)?.removeView(this)
    }
}

/**
 * Banner that resolves ad type (mraid, vast etc) automatically.
 * @param creativeType - bypasses creative resolving part
 * @param adm - ad markup; ad data to load and render.
 * @param dec - double end-card ad data to load and render.
 */
internal fun AggregatedBanner(
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    creativeType: CreativeType? = null,
    bid: Bid,
    options: AggregatedOptions,
    externalLinkHandler: ExternalLinkHandler,
    watermark: Watermark
): Banner<AggregatedAdShowListener> =
    com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.AggregatedBanner(
        context,
        customUserEventBuilderService,
        creativeType,
        bid,
        options,
        externalLinkHandler,
        watermark
    )
