package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.annotation.SuppressLint
import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.*
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType.*
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlin.time.Duration

@SuppressLint("ViewConstructor")
internal class AggregatedBanner(
    private val context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    creativeType: CreativeType?,
    private val bid: Bid,
    private val options: AggregatedOptions,
    val externalLinkHandler: ExternalLinkHandler,
    private val watermark: Watermark
) : Banner<AggregatedAdShowListener>(context) {
    private val TAG = "AggregatedBanner"
    init {
        tag = "MolocoAggregatedBannerView"
    }

    override var creativeType: CreativeType? = creativeType
        private set

    override var adShowListener: AggregatedAdShowListener? = null
        set(value) {
            field = value

            vastBanner
                ?.let { vastBanner -> vastBanner.adShowListener = value }
                ?: run { (mraidBanner ?: staticBanner)?.adShowListener = value }
        }

    private var vastBanner: Banner<VastAdShowListener>? = null
    private var mraidBanner: Banner<AdShowListener>? = null
    private var staticBanner: Banner<AdShowListener>? = null

    private val bannerImpl: Banner<*>?
        get() = vastBanner ?: mraidBanner ?: staticBanner

    override fun destroy() {
        scope.launch {
            super.destroy()
            bannerImpl?.destroy()
        }
    }

    override val adLoader: AdLoad = object : AdLoad {

        private val _isLoaded = MutableStateFlow(false)
        override val isLoaded = _isLoaded.asStateFlow()

        override fun load(timeout: Duration, listener: AdLoad.Listener?) {
            scope.launch {
                prepareBanner()
                // TODO: Banner should never be null at this point. Refactor to make it cleaner
                bannerImpl?.load(timeout, object : AdLoad.Listener {
                    override fun onLoad() {
                        listener?.onLoad()
                    }

                    override fun onLoadTimeout(timeoutError: AdLoadTimeoutError) {
                        when(this@AggregatedBanner.creativeType) {
                            VAST -> {
                                listener?.onLoadTimeout(AdLoadTimeoutError.VAST_BANNER_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                            }
                            MRAID -> {
                                listener?.onLoadTimeout(AdLoadTimeoutError.MRAID_BANNER_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                            }
                            STATIC -> {
                                listener?.onLoadTimeout(AdLoadTimeoutError.STATIC_BANNER_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                            }
                            null -> { MolocoLogger.error(TAG, "creativeType is null") }
                        }
                    }

                    override fun onLoadError(internalError: MolocoAdSubErrorType) {
                        listener?.onLoadError(internalError)
                    }
                })
            }
        }

        // TODO. Duplicate of AggregatedFullscreenAd::prepareAd()
        private suspend fun prepareBanner() {
            MolocoLogger.debug(TAG, "Preparing banner")
            // Possibly heavy stuff.
            val crType = this@AggregatedBanner.creativeType ?: withContext(DispatcherProvider().default) {
                CreativeTypeResolver.resolve(bid.adm).also {
                    this@AggregatedBanner.creativeType = it
                }
            }

            when (crType) {
                VAST -> vastBanner = VastBanner(context, customUserEventBuilderService, bid, options.vastOptions, externalLinkHandler, false /* No business use case to stream for banners (Figure out?) */)
                MRAID -> mraidBanner = MraidBanner(context, bid.adm, options.mraidOptions, externalLinkHandler, watermark)
                STATIC -> staticBanner = StaticBanner(context, customUserEventBuilderService, bid.adm, options.staticOptions, externalLinkHandler, watermark)
            }

            bannerImpl?.isLoaded?.onEach {
                _isLoaded.value = it
            }?.launchIn(scope)

            bannerImpl?.isAdDisplaying?.onEach {
                _isAdDisplaying.value = it
            }?.launchIn(scope)

            // Update created banner's show listener.
            adShowListener = adShowListener
        }
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying = _isAdDisplaying.asStateFlow()

    override fun prepareAdViewForDisplay() {
        adView = bannerImpl
    }
}
