package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.content.Context
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.FullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toMraidFullscreenError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidActivity
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidBaseAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidPlacementType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlin.time.Duration

internal fun MraidFullscreenAd(
    context: Context,
    bid: Bid,
    externalLinkHandler: ExternalLinkHandler,
    watermark: Watermark,
    decLoader: DECLoader
): FullscreenAd<AdShowListener, AdWebViewOptions> = MraidFullscreenAdImpl(
    context,
    externalLinkHandler,
    watermark,
    bid,
    decLoader
)

private class MraidFullscreenAdImpl(
    private val context: Context,
    externalLinkHandler: ExternalLinkHandler,
    private val watermark: Watermark,
    private val bid: Bid,
    decLoader: DECLoader,
) : FullscreenAd<AdShowListener, AdWebViewOptions> {

    override val creativeType = CreativeType.MRAID

    private val scope = CoroutineScope(DispatcherProvider().main)

    private val ad = object : MraidBaseAd(
        context,
        bid.adm,
        MraidPlacementType.Interstitial,
        onClick = { adShowListener?.onClick() },
        onError = this::onError,
        expandViewOptions = AdWebViewOptions(),
        externalLinkHandler,
        watermark = watermark
    ) {
        override fun closeFullscreenAdRepresentation() {
            super.closeFullscreenAdRepresentation()
            _isAdDisplaying.value = false
        }
    }

    private val mraidAdLoader = MraidAdLoad(scope, ad, bid, decLoader)

    private var adLoadListener: AdLoad.Listener? = null
    private var adShowListener: AdShowListener? = null

    override val isLoaded by mraidAdLoader::isLoaded

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        adLoadListener = listener
        mraidAdLoader.load(timeout, listener)
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> = _isAdDisplaying

    private val _isAdForciblyClosed = MutableStateFlow(false)
    override val isAdForciblyClosed: StateFlow<Boolean> = _isAdForciblyClosed

    override fun show(options: AdWebViewOptions, listener: AdShowListener?) {
        adShowListener = listener

        showCalled = true
        val adData = when(val mraidAdResult = mraidAdLoader.mraidLoadResult) {
            is Result.Failure -> {
                val error = mraidAdResult.value
                listener?.onShowError(error)
                return
            }
            is Result.Success -> {
                mraidAdResult.value
            }
        }

        if (MraidActivity.show(adData, ad.mraidFullscreenController, context, options, watermark)) {
            _isAdDisplaying.value = true
        } else {
            listener?.onShowError(MraidAdError.MRAID_FULLSCREEN_AD_ACTIVITY_SHOW_FAILED_ERROR)
        }
    }

    private var showCalled = false

    private fun onError(error: MraidAdError) {
        if (showCalled) {
            adShowListener?.onShowError(error.toMraidFullscreenError())
        } else {
            adLoadListener?.onLoadError(error.toMraidFullscreenError())
        }
    }

    override fun destroy() {
        scope.cancel()
        ad.destroy()
        // TODO. Refactor. Take Vast/Static implementation as example: suspendable show method wrapped with try finally block
        _isAdDisplaying.value = false
    }
}
