package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.content.Context
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.FullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toStaticFullscreenError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.StaticAdActivity
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.StaticWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.time.Duration

internal fun StaticFullscreenAd(
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    adm: String,
    externalLinkHandler: ExternalLinkHandler,
    watermark: Watermark
): FullscreenAd<AdShowListener, AdWebViewOptions> = StaticFullscreenAdImpl(
    context,
    customUserEventBuilderService,
    adm,
    externalLinkHandler,
    watermark
)

private class StaticFullscreenAdImpl(
    private val context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    adm: String,
    externalLinkHandler: ExternalLinkHandler,
    private val watermark: Watermark
) : FullscreenAd<AdShowListener, AdWebViewOptions> {

    override val creativeType = CreativeType.STATIC

    private val scope = CoroutineScope(DispatcherProvider().main)

    private val staticWebView: StaticWebView = StaticWebView(
        context,
        customUserEventBuilderService,
        externalLinkHandler,
    )

    private val staticAdLoader = StaticAdLoad(adm, scope, staticWebView)

    override val isLoaded by staticAdLoader::isLoaded

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        staticAdLoader.load(timeout, listener)
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> by lazy {
        // Added WebView's "isPageFinished" event check for static ads to prevent impression discrepancies,
        // between notifnt and html embedded trackers.
        _isAdDisplaying.combine(staticWebView.isPageFinished) { isAdDisplaying, isPageFinished ->
            isAdDisplaying && isPageFinished
        }.stateIn(scope, SharingStarted.Eagerly, false)
    }

    private val _isAdForciblyClosed = MutableStateFlow(false)
    override val isAdForciblyClosed: StateFlow<Boolean> = _isAdForciblyClosed

    override fun show(options: AdWebViewOptions, listener: AdShowListener?) {
        scope.launch {
            val eventHandlers = listOf(
                launch {
                    val error = staticWebView.unrecoverableError.first { it != null }
                    error?.let {
                        listener?.onShowError(it.toStaticFullscreenError())
                    }
                },
                launch {
                    staticWebView.clickthroughEvent.collect {
                        listener?.onClick()
                    }
                }
            )

            try {
                _isAdDisplaying.value = true
                StaticAdActivity.show(context, staticWebView, ::onClose, options, watermark)
            } finally {
                eventHandlers.onEach { it.cancel() }
                _isAdDisplaying.value = false
            }
        }
    }

    override fun destroy() {
        scope.cancel()
        staticWebView.destroy()
    }

    private fun onClose() {
        _isAdDisplaying.value = false
    }
}
