package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.mtid
import com.moloco.sdk.internal.toXenossDEC
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdLoadError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastAdLoader
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Ad
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.Job
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.async
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch
import kotlinx.coroutines.withTimeout
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration

internal class VastAdLoad(
    private val bid: Bid,
    private val scope: CoroutineScope,
    private val loadVast: VastAdLoader,
    private val decLoader: DECLoader,
    private val isStreamingEnabled: Boolean,
) : AdLoad {
    var vastAdLoadResult: Result<Ad, MolocoAdSubErrorType> = Result.Failure(VastAdLoadError.VAST_AD_LOAD_INCOMPLETE_ERROR)

    private val _isLoaded = MutableStateFlow(false)
    override val isLoaded = _isLoaded.asStateFlow()

    private var loadJob: Job? = null

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        // If streaming is enabled, we will load the ad in a streaming manner.
        // We will wait until the timeout for the ad to download the ad media
        // and if it has downloaded the media partially within the timeout, we will
        // consider that as successful ad load.
        if (isStreamingEnabled) {
            streamedLoad(timeout, listener)
        } else {
            fullLoad(timeout, listener)
        }
    }

    private fun fullLoad(timeout: Duration, listener: AdLoad.Listener?) {
        loadJob?.cancel()
        loadJob = scope.launch {
            // Ad is already loaded.
            if (vastAdLoadResult is Result.Success) {
                listener?.onLoad()
                return@launch
            }

            val vastAdDeferred = async {
                withTimeout(timeout) {
                    loadVast(bid.adm, mtid = bid.mtid(), false)
                }
            }

            // Presence/precaching of DEC is optional.
            val dec = bid.ext?.player?.dec?.toXenossDEC()
            val decDeferred = async {
                withTimeoutOrNull(timeout) {
                    dec?.let {
                        decLoader.load(it, bid.ext?.mtid)
                    }
                }
                // fallback in case timed out; still use original dec object.
                    ?: dec
            }

            val adLoadResult = try {
                vastAdDeferred.await()
            } catch (e: TimeoutCancellationException) {
                MolocoLogger.debug(TAG, "main VAST ad didn't load due to timeout")
                // if main ad didn't load in time, there's no need to wait for DEC.
                decDeferred.cancel()
                // Notify ad load timeout.
                val timeoutError = AdLoadTimeoutError.VAST_AD_LOAD_INTERNAL_TIMEOUT_ERROR
                vastAdLoadResult = Result.Failure(timeoutError)
                listener?.onLoadTimeout(timeoutError)
                return@launch
            }

            when(adLoadResult) {
                is Result.Success -> {
                    val vastAd = adLoadResult.value.copy(dec = decDeferred.await())
                    vastAdLoadResult = Result.Success(vastAd)
                    _isLoaded.value = true
                    listener?.onLoad()
                }

                is Result.Failure -> {
                    // Ad didn't load for other than timeout reasons.
                    handleFailure(decDeferred, listener, adLoadResult.value)
                }
            }
        }
    }

    private fun streamedLoad(timeout: Duration, listener: AdLoad.Listener?) {
        loadJob?.cancel()
        loadJob = scope.launch {
            // Ad is already loaded.
            if (vastAdLoadResult is Result.Success) {
                listener?.onLoad()
                return@launch
            }

            // Prepares the VAST ad (Does not actually trigger downloading the media files).
            val vastAdDeferred = loadVast(bid.adm, mtid = bid.mtid(), true)
            val dec = bid.ext?.player?.dec?.toXenossDEC()
            // Presence/precaching of DEC is optional.
            val decDeferred = async {
                withTimeoutOrNull(timeout) {
                    dec?.let {
                        decLoader.load(it, bid.ext?.mtid)
                    }
                }
                // fallback in case timed out; still use original dec object.
                    ?: dec
            }

            when(vastAdDeferred) {
                is Result.Success -> {
                    when(val result = loadVast.waitForAdLoadToStart(vastAdDeferred.value, timeout)) {
                        is Result.Success -> {
                            vastAdLoadResult = Result.Success(vastAdDeferred.value.copy(dec = decDeferred.await()))
                            _isLoaded.value = true
                            listener?.onLoad()
                        }
                        is Result.Failure -> {
                            MolocoLogger.debug(TAG, "main VAST ad didn't load due to failure or timeout")
                            if (result.value.isTimeoutError()) {
                                handleTimeout(decDeferred, listener, result.value)
                            } else {
                                handleFailure(decDeferred, listener, result.value)
                            }

                            return@launch
                        }
                    }
                }

                is Result.Failure -> {
                    handleFailure(decDeferred, listener, vastAdDeferred.value)
                    return@launch
                }
            }
        }
    }

    private fun handleTimeout(decDeferred: Deferred<DEC?>,
                              listener: AdLoad.Listener?,
                              error: MolocoAdSubErrorType) {
        // if main ad didn't load in time, there's no need to wait for DEC.
        decDeferred.cancel()
        val timeoutError = AdLoadTimeoutError.VAST_AD_LOAD_INTERNAL_TIMEOUT_ERROR
        vastAdLoadResult = Result.Failure(error)
        listener?.onLoadTimeout(timeoutError)
    }

    private fun handleFailure(decDeferred: Deferred<DEC?>,
                              listener: AdLoad.Listener?,
                              error: MolocoAdSubErrorType) {
        // Ad didn't load for other than timeout reasons.
        MolocoLogger.error(TAG, "Vast AD failed to load: $error")
        // if main ad didn't load, there's no need to wait for DEC.
        decDeferred.cancel()
        // Notify general ad load error.
        vastAdLoadResult = Result.Failure(error)
        listener?.onLoadError(error)
    }
}

private const val TAG = "VastAdLoad"
