package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.FullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdShowError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastActivity
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastAdLoader
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdControllerEvent
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlin.time.Duration

internal fun VastFullscreenAd(
    context: Context,
    bid: Bid,
    vastMediaStreamingEnabled: Boolean,
    watermark: Watermark,
    loadVast: VastAdLoader = VastAdLoader(context),
    decLoader: DECLoader = SdkObjectFactory.AdLoadModule.decLoader
): FullscreenAd<VastAdShowListener, VastOptions> = VastFullscreenAdImpl(
    context,
    bid,
    loadVast,
    decLoader,
    vastMediaStreamingEnabled,
    watermark
)

private class VastFullscreenAdImpl(
    private val context: Context,
    bid: Bid,
    loadVast: VastAdLoader,
    decLoader: DECLoader,
    vastMediaStreamingEnabled: Boolean,
    private val watermark: Watermark
) : FullscreenAd<VastAdShowListener, VastOptions> {

    override val creativeType = CreativeType.VAST

    private val scope = CoroutineScope(DispatcherProvider().main)

    private val vastAdLoader = VastAdLoad(bid, scope, loadVast, decLoader, vastMediaStreamingEnabled)

    override val isLoaded by vastAdLoader::isLoaded

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        vastAdLoader.load(timeout, listener)
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> = _isAdDisplaying

    private val _isAdForciblyClosed = MutableStateFlow(false)
    override val isAdForciblyClosed: StateFlow<Boolean> = _isAdForciblyClosed

    override fun show(options: VastOptions, listener: VastAdShowListener?) {
        scope.launch {
            val vastAd = when(val vastAdResult = vastAdLoader.vastAdLoadResult) {
                is Result.Failure -> {
                    val error = vastAdResult.value
                    listener?.onShowError(error)
                    return@launch
                }
                is Result.Success -> {
                    vastAdResult.value
                }
            }

            if (!vastAd.linear.localMediaResource.exists()) {
                MolocoLogger.info("VastFullscreenAdImpl", "VAST ad media file does not exist")
                listener?.onShowError(VastAdShowError.VAST_AD_EXOPLAYER_SET_MEDIA_FILE_NOT_EXISTS_ERROR)
                return@launch
            }

            try {
                VastActivity.showAd(vastAd, context, options, ::onClose, watermark) { event ->
                    onAdControllerEvent(event, listener)
                }
            } finally {
                _isAdDisplaying.value = false
            }
        }
    }

    /**
     * Handles the VAST ad controller events and notifies the listener accordingly.
     *
     * This function receives ad controller events and triggers corresponding actions, such as notifying the listener
     * about ad display progress, completion, errors, clicks, etc.
     *
     * @param event The ad controller event indicating various stages of ad display, such as start, completion, error, etc.
     * @param listener The listener to notify about the ad controller events. It allows the caller to receive callbacks
     *                 regarding the status of the ad display, such as errors, completions, clicks, etc.
     */
    private fun onAdControllerEvent(event: AdControllerEvent, listener: VastAdShowListener?){
        when (event) {
            AdControllerEvent.LinearDisplayStarted -> onShow()
            AdControllerEvent.CompanionDisplayStarted -> onShow()
            AdControllerEvent.DECDisplayStarted -> onShow()
            AdControllerEvent.Skip -> listener?.onVastCompletionStatus(skipped = true)
            AdControllerEvent.Complete -> listener?.onVastCompletionStatus(
                skipped = false
            )
            AdControllerEvent.ClickThrough -> listener?.onClick()
            is AdControllerEvent.Error -> listener?.onShowError(event.error)
            AdControllerEvent.Replay -> { /* no-op */ }
            AdControllerEvent.Dismiss -> { /* no-op */ }
        }
    }

    /**
     * Called from system onDestroy. Doesn't get called if normal ad close
     */
    private fun onClose() {
        _isAdForciblyClosed.value = true
        _isAdDisplaying.value = false
    }

    private fun onShow() {
        _isAdDisplaying.value = true
    }

    override fun destroy() {
        scope.cancel()
    }
}
