package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.error.ErrorMetadata
import com.moloco.sdk.internal.error.ErrorReportingService
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.ConnectivityService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.MediaStreamListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.MediaStreamStatus
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.NotStarted
import io.ktor.client.HttpClient
import io.ktor.client.plugins.retry
import io.ktor.client.request.get
import io.ktor.client.request.headers
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsChannel
import io.ktor.client.statement.request
import io.ktor.http.HttpHeaders
import io.ktor.http.contentLength
import io.ktor.util.cio.writeChannel
import io.ktor.utils.io.copyAndClose
import io.ktor.utils.io.core.isEmpty
import io.ktor.utils.io.core.readBytes
import kotlinx.coroutines.withContext
import java.io.File
import kotlin.math.min

private const val MAX_HTTP_RETRIES = 10
private const val WAIT_FOR_NETWORK = 5000L
internal const val CONTENT_RANGE_NOT_AVAILABLE = "CONTENT_RANGE_NOT_AVAILABLE"
internal interface ChunkedMediaDownloader {
    suspend fun downloadMedia(
        url: String,
        dstFile: File,
        mtid: String,
        listener: MediaStreamListener ?= null,
    ): MediaCacheRepository.Result

    fun isDownloaded(file: File): Boolean
    fun isPartiallyDownloaded(file: File): Boolean
}

/**
 * Downloads the media file in chunks
 */
internal class ChunkedMediaDownloaderImpl(
    private val mediaConfig: MediaConfig,
    private val connectivityService: ConnectivityService,
    private val errorReportingService: ErrorReportingService,
    private val httpClient: HttpClient,
) : ChunkedMediaDownloader {
    private val TAG = "ChunkedMediaDownloader"

    override fun isDownloaded(file: File): Boolean {
        return file.exists() && !rangeFile(file).exists()
    }

    override fun isPartiallyDownloaded(file: File): Boolean {
        return file.exists() && rangeFile(file).exists()
    }

    override suspend fun downloadMedia(
        url: String,
        dstFile: File,
        mtid: String,
        listener: MediaStreamListener?,
    ): MediaCacheRepository.Result = withContext(DispatcherProvider().io) {
        var previousBytes: Long
        var maxRange = previousRange(dstFile)?.split("/")?.last()?.toInt() ?: Int.MAX_VALUE
        var hasMoreData = true
        var remainingBytes = -1

        return@withContext try {
            MolocoLogger.info(TAG, "Fetching asset from network: $url")
            listener?.onStreamProgress(dstFile, NotStarted)
            previousBytes = dstFile.length().apply {
                MolocoLogger.info(TAG, "Previous tmpfile bytes: $this")
            }

            if (maxRange.toLong() == previousBytes) {
                MolocoLogger.info(TAG, "File already downloaded, skipping download")
                return@withContext onMediaDownloadComplete(dstFile, listener)
            }

            val previousEtag = previousEtag(dstFile)
            var chunk = 0
            while(hasMoreData) {
                val hasNetwork = connectivityService.waitForNetwork(WAIT_FOR_NETWORK)
                if (!hasNetwork) {
                    listener?.onStreamError(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.NoNetworkError))
                    return@withContext MediaCacheRepository.Result.Failure.NoNetworkError
                }

                chunk++
                MolocoLogger.info(
                    TAG,
                    "Making request to fetch chunk: $chunk for remainingBytes: $remainingBytes"
                )
                val response = makeRequest(url, previousBytes, maxRange, previousEtag)
                val result = mediaDownloadResult(dstFile, response, listener)
                if (result is MediaCacheRepository.Result.Failure) {
                    return@withContext result
                }
                processEtag(dstFile, response)

                MolocoLogger.info(TAG, "ResponseCode: ${response.status.value}, ${HttpHeaders.ContentLength}: ${response.contentLength()}")

                val contentRange = response.headers[HttpHeaders.ContentRange]
                if (contentRange != null) {
                    MolocoLogger.info(
                        TAG,
                        "Content range header is available, ${HttpHeaders.ContentRange}: $contentRange"
                    )
                    // range file should be persisted immediately before any chunk is written to disk
                    // This means that if the process dies before the chunk is persisted,
                    // the range file can be ahead of the chunk actually on disk, so always use the chunk size
                    // to determine most recent downloaded byte. Use the range file to know how much max to download in a range
                    processRange(dstFile, contentRange)
                    maxRange = contentRange.split("/").last().toInt() // Also the total size of the file (on server)
                    val contentLength = response.contentLength() ?: 0
                    val responseRange = contentRange.split("/").first()
                    val rangeEnd = when (responseRange.contains("-")) {
                        false -> maxRange
                        true -> responseRange.split("-").last().toInt()
                    }
                    remainingBytes = maxRange - rangeEnd - 1// Total file size on server - the latest byte offset in range header
                    MolocoLogger.info(TAG, "maxRange: $maxRange, Response contentLength: $contentLength")
                    hasMoreData = remainingBytes > 0 /* total > 0 && total < range */
                    previousBytes += contentLength

                    writeChunkToFile(dstFile, response)

                    listener?.onStreamProgress(dstFile, MediaStreamStatus.Progress(dstFile.length(), maxRange.toLong()))
                    if (hasMoreData) {
                        MolocoLogger.info(TAG, "Server has more data")
                    } else {
                        MolocoLogger.info(TAG, "Server does not have more data to send")
                    }
                } else {
                    MolocoLogger.warn(TAG, "${HttpHeaders.ContentRange} is not available")
                    errorReportingService.reportError(CONTENT_RANGE_NOT_AVAILABLE, ErrorMetadata(mtid = mtid))
                    downloadFullFile(dstFile, response)
                    hasMoreData = false
                }
            }

            onMediaDownloadComplete(dstFile, listener)
        } catch (e: Exception) {
            val error = exceptionToMediaResult(e)
            MolocoLogger.error(TAG, "Failed to fetch media from url: $url due to error: $error", e)
            listener?.onStreamError(MediaStreamStatus.Failure(error))
            error
        }
    }

    private fun mediaDownloadResult(dstFile: File, response: HttpResponse, listener: MediaStreamListener?): MediaCacheRepository.Result {
        return when (response.status.value) {
            in 400..499 -> {
                MolocoLogger.error(TAG, "Failed to fetch media from url: ${response.request.url}, status: ${response.status}")
                listener?.onStreamError(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.HttpClientError))
                MediaCacheRepository.Result.Failure.HttpClientError
            }
            in 500..599 -> {
                MolocoLogger.error(TAG, "Failed to fetch media from url: ${response.request.url}, status: ${response.status}")
                listener?.onStreamError(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.HttpServerError))
                MediaCacheRepository.Result.Failure.HttpServerError
            }
            else -> {
                MediaCacheRepository.Result.Success(dstFile)
            }
        }
    }

    private fun onMediaDownloadComplete(dstFile: File, listener: MediaStreamListener?): MediaCacheRepository.Result {
        // Before returning the file, we remove the etag file since we already have
        // the full file on disk
        removeEtag(dstFile)
        removeRange(dstFile)
        listener?.onStreamComplete(MediaStreamStatus.Complete(dstFile))
        return MediaCacheRepository.Result.Success(dstFile)
    }

    private suspend fun makeRequest(url: String, previousBytes: Long, maxRange: Int, previousEtag: String?): io.ktor.client.statement.HttpResponse {
        return httpClient.get(url) {
            retry {
                // If there are failures, then we will retry 10 times back to back
                // Why 10 and why back to back?
                // This is because we are trying to download a file in chunks and if there is a failure in one chunk, we will retry that chunk again
                maxRetries = MAX_HTTP_RETRIES
                delayMillis { _ -> 100L}
                retryOnException(MAX_HTTP_RETRIES, true)
                retryOnServerErrors(MAX_HTTP_RETRIES)
                modifyRequest {
                    MolocoLogger.info(
                        TAG,
                        "Retry attempt #${this.retryCount} for ${this.request.url}"
                    )
                }
            }

            headers {
                val range = "bytes=$previousBytes-${min(previousBytes + mediaConfig.chunkSize, maxRange.toLong())}"
                MolocoLogger.info(TAG, "Adding ${HttpHeaders.Range} header: $range")
                append(HttpHeaders.Range, range)

                // If the server supports ETag, then we can use it to check if the file has changed
                // If the file has changed then the server will send the whole file down instead of
                // chunks
                if (previousEtag != null) {
                    MolocoLogger.info(TAG, "Adding ${HttpHeaders.IfRange} header: $previousEtag")
                    append(HttpHeaders.IfRange, previousEtag)
                    MolocoLogger.info(TAG, "Adding ${HttpHeaders.ETag} header: $previousEtag")
                    append(HttpHeaders.ETag, previousEtag)
                }
            }
        }
    }

    private fun rangeFile(dstFile: File): File {
        return File(dstFile.parent, "${dstFile.name}.range")
    }

    private fun removeRange(dstFile: File) {
        rangeFile(dstFile).delete()
    }

    private fun processRange(dstFile: File, range: String) {
        rangeFile(dstFile).writeText(range)
    }

    private fun previousRange(dstFile: File): String? {
        val rangeFile = rangeFile(dstFile)
        return if (rangeFile.exists()) {
            rangeFile.readText()
        } else {
            null
        }
    }

    private fun etagFile(dstFile: File): File {
        return File(dstFile.parent, "${dstFile.name}.etag")
    }

    private fun removeEtag(dstFile: File) {
        etagFile(dstFile).delete()
    }

    private fun processEtag(dstFile: File, response: io.ktor.client.statement.HttpResponse) {
        val etag = response.headers[HttpHeaders.ETag]
        if (etag != null) {
            MolocoLogger.info(TAG, "${HttpHeaders.ETag}: $etag")
            etagFile(dstFile).writeText(etag)
        } else {
            MolocoLogger.warn(TAG, "No ${HttpHeaders.ETag} in header")
            removeEtag(dstFile)
        }
    }

    private fun previousEtag(dstFile: File): String? {
        val etagFile = etagFile(dstFile)
        return if (etagFile.exists()) {
            etagFile.readText()
        } else {
            null
        }
    }

    private suspend fun writeChunkToFile(dstFile: File, response: io.ktor.client.statement.HttpResponse) {
        val channel = response.bodyAsChannel()
        while (!channel.isClosedForRead) {
            val packet = channel.readRemaining(mediaConfig.chunkSize * 2L)
            while (!packet.isEmpty) {
                val bytes = packet.readBytes()
                dstFile.appendBytes(bytes)
                MolocoLogger.info(TAG, "dst file length: ${dstFile.length()} bytes")
            }
        }
    }

    private suspend fun downloadFullFile(dstFile: File, response: io.ktor.client.statement.HttpResponse) {
        MolocoLogger.info(TAG, "Range header not supported, downloading full file")
        if (dstFile.exists()) {
            MolocoLogger.info(TAG, "Deleting existing file and fully re-downloading it")
            dstFile.delete()
        }

        val writtenBytes = response.bodyAsChannel().copyAndClose(dstFile.writeChannel())
        MolocoLogger.info(TAG, "Downloaded full response: ${response.contentLength()} and saved to disk: $writtenBytes bytes, file size: ${dstFile.length()}")
    }
}