package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.Error
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.Result
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.MediaStreamListenerFlow
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.MediaStreamStatus
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.NotStarted
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.toMD5WithoutQueryParams
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import java.io.File
import java.util.concurrent.ConcurrentHashMap

internal const val MEDIA_CACHE_DIR = "com.moloco.sdk.xenoss.sdkdevkit.android.cache"

/**
 * Fetches and caches network media (or any other file/data in the current implementation).
 */
internal interface MediaCacheRepository {
    /**
     * Retrieves the media file from the cache or downloads it from the network.
     */
    suspend fun getMediaFile(url: String): Result

    /**
     * Retrieves the media file from the cache or downloads it from the network. The content is streamed
     * and not fetched at once. Use the stream status API to get the ongoing download status.
     *
     * @return [MediaStreamStatus] that represents the snapshot status of the download.
     * NotStarted if the download has not started yet, InProgress if it was already started or Complete if the download is finished.
     *
     */
    suspend fun streamMediaFile(url: String, mtid: String): MediaStreamStatus

    /**
     * @return [Flow] of [MediaStreamStatus] that represents the ongoing download status of the media file.
     */
    fun streamMediaFileStatus(url: String): Flow<MediaStreamStatus>
    fun streamMediaFileStatusSnapshot(url: String): MediaStreamStatus

     // TODO. Refactor? It's an implementation detail,
     //  it isn't supposed to be public and called externally.
    fun tryCleanup(): Job

    sealed class Result {
        class Success(val file: File) : Result()

        sealed class Failure : Result() {
            // Errors that can result from http calls
            object UnknownHostHttpError : Failure()
            object HttpServerError : Failure()
            object HttpClientError : Failure()
            object HttpDiskIOError : Failure()
            object HttpSocketError : Failure()
            object HttpDiskSecurityError : Failure()
            object HttpSslError : Failure()

            object NoNetworkError : Failure()
            object UnableToRenameTmpFileError : Failure()

            // Errors that can result from disk operations
            object FileNotCreatedDiskError : Failure()
            object FileNotCreatedSecurityError : Failure()
            object FileNotCreatedIOError : Failure()
            object FileNotCreatedUnknownError : Failure()

            // Timeout
            object MediaFetchTimeoutError : Failure()

            // General errors
            object UnknownMediaFetchError : Failure()
            object InvalidUrlError : Failure()
            object NotFound : Failure()
        }

    }
}

internal typealias CacheDirResult = Result<File, MediaStreamStatus.Failure>


// TODO. IMPORTANT. Cache cleanup once in a while (WorkManager or async task).
// TODO: Check for concurrency safety if clean up is hooked up to ensure download and clean up are not triggered at same time
internal class MediaCacheRepositoryImpl(
    private val mediaConfig: MediaConfig,
    private val legacyMediaDownloader: LegacyMediaDownloader,
    private val chunkedMediaDownloader: ChunkedMediaDownloader,
    private val mediaCacheLocationProvider: MediaCacheLocationProvider,
) : MediaCacheRepository {

    private val scope = CoroutineScope(DispatcherProvider().io)
    private val urlMutexMap = ConcurrentHashMap<String, Mutex>()
    private val inProgressStreamedDownloads = HashSet<String>()
    private val inProgressStreamDownloadStatusFlows = ConcurrentHashMap<String, MediaStreamListenerFlow>()


    override suspend fun getMediaFile(url: String): MediaCacheRepository.Result =
        withContext(DispatcherProvider().io) {
            if (url.isEmpty()) return@withContext MediaCacheRepository.Result.Failure.InvalidUrlError

            val mutex = urlMutexMap.getOrPut(url) { Mutex() }
            mutex.withLock {
                val result = try {
                    val cacheDir = when (val cacheDirResult = getCacheDir()) {
                        is Result.Failure -> {
                            MolocoLogger.warn(
                                TAG,
                                "Failed to retrieve storageDir with error code: ${cacheDirResult.value.failureCode}"
                            )
                            return@withContext when (cacheDirResult.value.failureCode) {
                                FILE_CREATE_SECURITY_ERROR -> MediaCacheRepository.Result.Failure.FileNotCreatedSecurityError
                                FILE_CREATE_STORAGE_DIRECTORY_IO_ERROR -> MediaCacheRepository.Result.Failure.FileNotCreatedIOError
                                FILE_CREATE_STORAGE_DIRECTORY_NOT_AVAILABLE_ERROR -> MediaCacheRepository.Result.Failure.FileNotCreatedDiskError
                                else -> MediaCacheRepository.Result.Failure.FileNotCreatedUnknownError
                            }
                        }

                        is Result.Success -> cacheDirResult.value
                    }

                    // The backend sometimes appends query parameters to the media url to track additional
                    // information. We need to hash the url without the query parameters to create a unique
                    // file name for the media file.
                    val urlMD5Hash = url.toMD5WithoutQueryParams()
                    MolocoLogger.info(TAG, "Created md5 hash: $urlMD5Hash for url: $url")

                    val dstFile = File(cacheDir, urlMD5Hash)
                    if (dstFile.exists()) {
                        // This check is to handle the case where streaming config is enabled and turned off
                        if (chunkedMediaDownloader.isPartiallyDownloaded(dstFile)) {
                            MolocoLogger.info(
                                TAG,
                                "Media file was partially downloaded by ChunkedMediaDownloader. Deleting the file and redownloading"
                            )
                            dstFile.delete()
                        } else {
                            MolocoLogger.info(TAG, "Found asset in cache: $url")
                            return@withContext MediaCacheRepository.Result.Success(dstFile)
                        }
                    }

                    val tmpFile = File(cacheDir, urlMD5Hash.withTempPrefix())
                    MolocoLogger.debug(
                        TAG,
                        "Asset not found in cache. Downloading to tmp file[already exists == ${tmpFile.exists()}]"
                    )
                    // To prevent appending data to existing one and rendering file corrupted.
                    if (tmpFile.exists()) tmpFile.delete()

                    when (val result = legacyMediaDownloader.downloadMedia(url, tmpFile)) {
                        is MediaCacheRepository.Result.Success -> {
                            // Kind of "atomic" file copy mechanism to prevent file corruption
                            // in case of networking problems or killing process.
                            MolocoLogger.debug(TAG, "Renaming tmp file to dst file")
                            if (!tmpFile.renameTo(dstFile)) {
                                MolocoLogger.info(
                                    TAG,
                                    "Renaming to dst file failed, dstFile exists: ${dstFile.exists()}"
                                )
                                return@withContext MediaCacheRepository.Result.Failure.UnableToRenameTmpFileError
                            }
                            return@withContext MediaCacheRepository.Result.Success(dstFile)
                        }

                        else -> {
                            return@withContext result
                        }
                    }
                } catch (e: Exception) {
                    MolocoLogger.error(TAG, "Failed to fetch media from url: $url", e)
                    exceptionToMediaResult(e)
                }

                return@withContext result
            }
        }

    override suspend fun streamMediaFile(url: String, mtid: String): MediaStreamStatus =
        withContext(DispatcherProvider().io) {
            MolocoLogger.info(TAG, "Streaming media for: $url")
            if (url.isEmpty()) return@withContext MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.InvalidUrlError)
            val mutex = urlMutexMap.getOrPut(url) { Mutex() }
            mutex.withLock {
                val cacheDir = when(val cacheDirResult = cacheDirResult()) {
                    is Result.Failure -> return@withContext cacheDirResult.value
                    is Result.Success -> cacheDirResult.value
                }

                val dstFile = destinationFile(url, cacheDir)
                MolocoLogger.info(TAG, "Going to download the media file to location: ${dstFile.absolutePath}")

                val listener = inProgressStreamDownloadStatusFlows[url]

                // 1. It is partially downloaded and downloader is still downloading it actively
                if (inProgressStreamedDownloads.contains(url)) {
                    MolocoLogger.info(TAG, "Media file is already being downloaded, so returning in progress status for url: $url")
                    return@withContext listener?.lastStreamStatus ?: MediaStreamStatus.InProgress(dstFile, NotStarted)
                }

                // 2. It is fully downloaded
                // Media file is either fully downloaded
                if (chunkedMediaDownloader.isDownloaded(dstFile)) {
                    MolocoLogger.info(TAG, "Media file is already fully downloaded, so returning complete status for url: $url")
                    return@withContext MediaStreamStatus.Complete(dstFile)
                }
                // Media file is not being downloaded
                // actively (was partially downloaded in previous app session or not present in cache)

                MolocoLogger.info(TAG, "Media file needs to be downloaded: $url")

                // Downloader is not actively downloading,
                // track the download and kick off a download scope
                inProgressStreamedDownloads.add(url)
                val mediaListener = inProgressStreamDownloadStatusFlows.getOrPut(url) {
                    MediaStreamListenerFlow(MediaStreamStatus.InProgress(dstFile, NotStarted))
                }
                scope.launch {
                    chunkedMediaDownloader.downloadMedia(url, dstFile, mtid, mediaListener)
                    inProgressStreamedDownloads.remove(url)
                    inProgressStreamDownloadStatusFlows.remove(url)
                    return@launch
                }

                return@withContext mediaListener.lastStreamStatus
            }
        }

    override fun streamMediaFileStatus(url: String): Flow<MediaStreamStatus> {
        val cacheDir = when(val cacheDirResult = cacheDirResult()) {
            is Result.Failure -> return flow {cacheDirResult.value}
            is Result.Success -> cacheDirResult.value
        }

        MolocoLogger.info(TAG, "Collecting status for media file: $url")

        val dstFile = destinationFile(url, cacheDir)

        if (dstFile.exists() && chunkedMediaDownloader.isDownloaded(dstFile)) {
            MolocoLogger.info(TAG, "Media file is already fully downloaded, so returning complete status for url: $url")
            return flow { emit(MediaStreamStatus.Complete(dstFile)) }
        }

        MolocoLogger.info(TAG, "Media file needs to be downloaded: $url")
        // If the file has not started downloading or is in progress, return the existing stream status flow or create a new one
        val flow = inProgressStreamDownloadStatusFlows.getOrPut(url) {
            MolocoLogger.info(TAG, "Download has not yet started for: $url")
            MediaStreamListenerFlow(MediaStreamStatus.InProgress(dstFile, NotStarted))
        }.streamStatusFlow

        return flow
    }

    override fun streamMediaFileStatusSnapshot(url: String): MediaStreamStatus {
        val cacheDir = when(val cacheDirResult = cacheDirResult()) {
            is Result.Failure -> return cacheDirResult.value
            is Result.Success -> cacheDirResult.value
        }
        val dstFile = destinationFile(url, cacheDir)

        if (dstFile.exists() && chunkedMediaDownloader.isDownloaded(dstFile)) {
            MolocoLogger.info(TAG, "Media file is already fully downloaded, so returning complete status for url: $url")
            return MediaStreamStatus.Complete(dstFile)
        }

        return inProgressStreamDownloadStatusFlows[url]?.lastStreamStatus ?: MediaStreamStatus.InProgress(dstFile, NotStarted)
    }

    private fun destinationFile(url: String, cacheDir: File): File {
        // The backend sometimes appends query parameters to the media url to track additional
        // information. We need to hash the url without the query parameters to create a unique
        // file name for the media file.

        val urlMD5Hash = url.toMD5WithoutQueryParams()
        MolocoLogger.info(TAG, "Created md5 hash: $urlMD5Hash for url: $url")
        return File(cacheDir, urlMD5Hash)
    }

    private fun cacheDirResult(): CacheDirResult {
        return when(val cacheDirResult = getCacheDir()) {
            is Result.Failure -> {
                MolocoLogger.warn(TAG, "Failed to retrieve storageDir with error code: ${cacheDirResult.value.failureCode}")
                return when(cacheDirResult.value.failureCode) {
                    FILE_CREATE_SECURITY_ERROR -> Result.Failure(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.FileNotCreatedSecurityError))
                    FILE_CREATE_STORAGE_DIRECTORY_IO_ERROR -> Result.Failure(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.FileNotCreatedIOError))
                    FILE_CREATE_STORAGE_DIRECTORY_NOT_AVAILABLE_ERROR -> Result.Failure(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.FileNotCreatedDiskError))
                    else -> Result.Failure(MediaStreamStatus.Failure(MediaCacheRepository.Result.Failure.FileNotCreatedUnknownError))
                }
            }
            is Result.Success -> Result.Success(cacheDirResult.value)
        }
    }

    /**
     * Try to create a file in the external cache directory,
     * then internal cache directory and if both fails, then null
     */
    @VisibleForTesting
    internal fun getCacheDir(): Result<File, Error> =
        when(val dir = mediaCacheLocationProvider.externalCacheDirForStorage()) {
            is Result.Failure -> mediaCacheLocationProvider.internalCacheDirForStorage()
            is Result.Success -> dir
        }

    private val cleanupScope = CoroutineScope(DispatcherProvider().io)
    private var cleanupJob: Job? = null

    // TODO. File read/write synchronization of some sort.
    override fun tryCleanup(): Job {
        val cleanUpjob = cleanupJob

        if (cleanUpjob != null && cleanUpjob.isActive) {
            return cleanUpjob
        }

        val newCleanUpJob = cleanupScope.launch {
            when(val cacheDir = mediaCacheLocationProvider.externalCacheDirForStorage()) {
                is Result.Success -> performCleanUp(cacheDir.value)
                is Result.Failure -> MolocoLogger.error(TAG, "Failed to cleanup external cache directory")
            }

            when(val cacheDir = mediaCacheLocationProvider.internalCacheDirForStorage()) {
                is Result.Success -> performCleanUp(cacheDir.value)
                is Result.Failure -> MolocoLogger.error(TAG, "Failed to cleanup internal cache directory")
            }
        }
        cleanupJob = newCleanUpJob
        return newCleanUpJob
    }

    private fun performCleanUp(cacheDir: File) {
        val dirSizeInBytes = try {
            cacheDir.walkTopDown().map { it.length() }.sum()
        } catch (e: Exception) {
            MolocoLogger.error(TAG, e.toString(), e)
            return
        }

        if (dirSizeInBytes < mediaConfig.mediaCacheDiskCleanUpLimit) {
            return
        }

        try {
            cacheDir.deleteRecursively()
        } catch (e: Exception) {
            MolocoLogger.error(TAG, e.toString(), e)
        }
    }

    companion object {
        private val TAG = "MediaCacheRepository"

        private const val TEMP = "TEMP"

        private fun String.withTempPrefix() = "${this}$TEMP"
    }
}
