package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import android.os.Bundle
import android.view.ViewGroup
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.runtime.Composable
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.closeDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.decDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putCloseDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putDECDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultAdCloseCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.mraidAdPlaylistController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultAdSkipCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview.AdWebViewRenderer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview.MraidAdWebViewScreen
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import java.lang.ref.WeakReference

internal class MraidActivity : ComponentActivity() {

    private val TAG = "MraidActivity"
    private val scope = CoroutineScope(DispatcherProvider().main)
    private val customUserEventBuilderService by lazy { SdkObjectFactory.UserTracking.customUserEventBuilderFactory }

    private fun startListeningToMraidOrientationEvent(
        expectedOrientation: StateFlow<MraidJsCommand.SetOrientationProperties?>,
    ) {
        // Speed up things a lil bit ot avoid screen rotation jitter..
        setOrientation(expectedOrientation.value)
        // TODO. What about when user change orientation manually
        //  and mraid gonna send a different orientation? Edge case but retarded...
        expectedOrientation.onEach(::setOrientation).launchIn(scope)
    }

    private fun setOrientation(orientationCommand: MraidJsCommand.SetOrientationProperties?) {
        // TODO. Orientation locking.
        //  Let's ignore it for now.
        // it.allowOrientationChange ... SCREEN_ORIENTATION_LOCKED ...

        orientationCommand
            ?.forceOrientation
            ?.toActivityOrientation()
            ?.let { requestedOrientation = it }
    }

    private fun MraidOrientation.toActivityOrientation() = when (this) {
        MraidOrientation.Portrait -> ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        MraidOrientation.Landscape -> ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        MraidOrientation.None -> null // Nothing...
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        MraidActivityDataHolder.weakActivity = this

        val closeButton = MraidActivityDataHolder.closeButton
        val mraidRenderer = MraidActivityDataHolder.MraidRenderer
        if (mraidRenderer == null) {
            MolocoLogger.error(TAG, "can't display ad: MraidRenderer is missing")
            finish()
            return
        }

        val mraidContentController = MraidActivityDataHolder.weakController

        if (mraidContentController == null) {
            MolocoLogger.error(TAG, "can't display ad: mraid controller is missing")
            finish()
            return
        }

        val adPlaylistController = MraidActivityDataHolder.adData?.let {
            mraidAdPlaylistController(
                mraidAdData = it,
                externalLinkHandler = SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                context = this,
                mraidFullscreenContentController = mraidContentController,
                decGoNextActionDelaySeconds = intent.decDelaySeconds,
                customUserEventBuilderService = customUserEventBuilderService
            )
        }

        if (adPlaylistController == null) {
            MolocoLogger.error(TAG, "can't display ad: mraid ad data is missing")
            finish()
            return
        }

        startListeningToMraidOrientationEvent(mraidContentController.expectedOrientation)

        adPlaylistController.event.onEach {
            CurrentMraidEvents.emit(it)
            if (it.isErrorOrDismiss) {
                adPlaylistController.destroy()
                finish()
            }
        }.launchIn(scope)

        setContent {
            MraidAdWebViewScreen(
                adViewModel = adPlaylistController,
                webView = mraidContentController.webView,
                closeDelaySeconds = intent.closeDelaySeconds,
                onButtonRendered = { /* NO OP: MRAID not supported */ },
                onClose = mraidContentController::onSkipOrClose,
                adWebViewRenderer = mraidRenderer,
                watermark = MraidActivityDataHolder.watermark,
                adCloseCountdownButton = closeButton(),
                adSkipCountdownButton = defaultAdSkipCountdownButton()
            )
        }

        adPlaylistController.show()
    }

    override fun onDestroy() {
        super.onDestroy()
        scope.cancel()
    }

    // "Weak ref" stuff should work fine for single activity instance, since there's only one fullscreen ad displayed at time.
    // If that's not enough or in case of some simultaneous multi ad display edge cases:
    // implement fullscreen ad manager with storage like map: <UUID_or_int_increment, Pair<MraidController, Activity>>
    // where bundle start activity intent contains the UUID_or_int_increment so you can match all needed components of the ad,
    // therefore it's easy to create multiple instances of the fullscreen ad.
    companion object {
        private val CurrentMraidEvents = MutableSharedFlow<AdControllerEvent>()

        fun show(
            adData: MraidAdData,
            controller: MraidFullscreenContentController,
            context: Context,
            options: AdWebViewOptions,
            watermark: Watermark?
        ): Boolean {
            if (!destroy(controller)) return false

            MraidActivityDataHolder.adData = adData
            MraidActivityDataHolder.watermark = watermark
            MraidActivityDataHolder.MraidRenderer = options.AdWebViewRenderer
            MraidActivityDataHolder.closeButton = options.decClose

            MraidActivityDataHolder.weakController = controller
            context.startActivity(
                Intent(context, MraidActivity::class.java).apply {
                    putCloseDelaySeconds(options.closeDelaySeconds)
                    putDECDelaySeconds(options.decGoNextActionDelaySeconds)
                    flags = Intent.FLAG_ACTIVITY_NEW_TASK
                }
            )

            return true
        }

        private val AdControllerEvent.isErrorOrDismiss: Boolean
            get() = this is AdControllerEvent.Error || this == AdControllerEvent.Dismiss

        // TODO. Check if controller is the same to avoid destroying not owned activities.
        //  Ugly but works. Will change when have time/mood
        //  to something like: suspend fun show() with destroy logic during scope cancellation.
        fun destroy(controller: MraidFullscreenContentController): Boolean =
            with(MraidActivityDataHolder) {
                val wc = weakController
                // Treat null as correct value just in case :)
                return if (wc == null || wc == controller) {
                    weakController = null

                    // As ugly as it gets: remove webview to avoid attach/detach issues for banners.
                    (wc?.webView?.parent as? ViewGroup)?.removeView(wc.webView)

                    MraidRenderer = null
                    adData = null
                    watermark = null

                    weakActivity?.finish()
                    weakActivity = null
                    true
                } else {
                    false
                }
            }
    }
}

private object MraidActivityDataHolder {

    private var _weakController = WeakReference<MraidFullscreenContentController?>(null)

    var weakController: MraidFullscreenContentController?
        get() = _weakController.get()
        set(value) {
            _weakController = WeakReference(value)
        }

    private var _weakActivity = WeakReference<Activity?>(null)

    var weakActivity: Activity?
        get() = _weakActivity.get()
        set(value) {
            _weakActivity = WeakReference(value)
        }

    var MraidRenderer: AdWebViewRenderer? = null
    var closeButton: @Composable () -> AdCountdownButton? = { defaultAdCloseCountdownButton() }
    var adData: MraidAdData? = null
    var watermark: Watermark? = null
}
