package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.webkit.WebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.companion.CompanionGoNextAction
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.launch

/**
 * A controller for managing MRAID fullscreen ad interactions.
 *
 * @property hasClickThrough Indicates whether the ad has a clickable component that leads to another page.
 * @property mraidBridge The bridge for communication between the JavaScript and the native layer.
 * @property closeFullscreenAdRepresentation A function to close the fullscreen ad representation.
 * @param scope The [CoroutineScope] in which asynchronous tasks are launched.
 */
internal class MraidFullscreenContentControllerImpl(
    override val hasClickThrough: Boolean,
    private val mraidBridge: MraidBridge,
    private val closeFullscreenAdRepresentation: () -> Unit,
    private val scope: CoroutineScope,
    goNextActionDelaySeconds: UInt
) : MraidFullscreenContentController {

    private val goNextActionHolder = CompanionGoNextAction(
        goNextActionDelaySeconds,
        scope
    )
    override val goNextAction by goNextActionHolder::goNextAction

    private val _event = MutableSharedFlow<MraidFullscreenControllerEvent>()
    override val event: Flow<MraidFullscreenControllerEvent> = _event
    private fun onEvent(event: MraidFullscreenControllerEvent) = scope.launch { _event.emit(event) }

    override val webView: WebView
        get() = mraidBridge.webView

    override val expectedOrientation: StateFlow<MraidJsCommand.SetOrientationProperties?> =
        mraidBridge.mraidJsCommands.mapNotNull {
            it as? MraidJsCommand.SetOrientationProperties
        }.stateIn(scope, SharingStarted.Eagerly, null)

    override fun onSkipOrClose() {
        onEvent(MraidFullscreenControllerEvent.SkipOrClose)
    }

    override fun onClickThrough(position: CustomUserEventBuilderService.UserInteraction.Position) {
        onEvent(MraidFullscreenControllerEvent.ClickThrough)
    }

    override fun destroy() {
        closeFullscreenAdRepresentation()
    }
}
