package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.nativead

import android.content.Context
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.Result
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastAdLoader
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.coroutineScope

// TODO. Asset preparation Failure default/overridable logic (if required and not loaded -- fail ... or custom logic)
internal suspend fun prepareNativeAssets(
    assets: List<NativeOrtbResponse.Asset>,
    context: Context,
): Result<PreparedNativeAssets, String> {
    val loadVast = lazy { VastAdLoader(context) }

    val groupedAssets = assets.groupBy { it.required }
    val requiredAssetsGroup = groupedAssets[true] ?: listOf()
    val optionalAssetsGroup = groupedAssets[false] ?: listOf()

    // Loading required assets firsts, fail everything immediately if any required asset doesn't get "prepared".
    val preparedRequiredAssets = try {
        coroutineScope {
            requiredAssetsGroup.map { asset ->
                async {
                    when (val result = prepareNativeAsset(asset, loadVast)) {
                        is Result.Failure -> throw Exception(result.value)
                        is Result.Success -> asset to result
                    }
                }
            }.awaitAll()
        }
    } catch (e: Exception) {
        return Result.Failure("required asset failed to prepare: $e")
    }

    val preparedOptionalAssets = coroutineScope {
        optionalAssetsGroup.map { asset ->
            async { asset to prepareNativeAsset(asset, loadVast) }
        }.awaitAll()
    }

    val data = mutableMapOf<Int, PreparedNativeAsset.Data>()
    val images = mutableMapOf<Int, PreparedNativeAsset.Image>()
    val titles = mutableMapOf<Int, PreparedNativeAsset.Title>()
    val videos = mutableMapOf<Int, PreparedNativeAsset.Video>()
    val failedAssets = mutableListOf<Pair<NativeOrtbResponse.Asset, String>>()

    for ((originAsset, preparedNativeAssetResult) in preparedRequiredAssets + preparedOptionalAssets) {
        when (preparedNativeAssetResult) {
            is Result.Failure -> failedAssets += originAsset to preparedNativeAssetResult.value
            is Result.Success -> when (val res = preparedNativeAssetResult.value) {
                is PreparedNativeAsset.Data -> data += res.originAsset.id to res
                is PreparedNativeAsset.Image -> images += res.originAsset.id to res
                is PreparedNativeAsset.Title -> titles += res.originAsset.id to res
                is PreparedNativeAsset.Video -> videos += res.originAsset.id to res
            }
        }
    }

    return Result.Success(PreparedNativeAssets(data, images, titles, videos, failedAssets))
}

private suspend fun prepareNativeAsset(
    asset: NativeOrtbResponse.Asset,
    loadVast: Lazy<VastAdLoader>
): Result<PreparedNativeAsset, String> = when (asset) {
    is NativeOrtbResponse.Asset.Data -> Result.Success(
        PreparedNativeAsset.Data(asset)
    )

    is NativeOrtbResponse.Asset.Image -> prepareImageAsset(
        asset
    )

    is NativeOrtbResponse.Asset.Title -> Result.Success(
        PreparedNativeAsset.Title(asset)
    )

    is NativeOrtbResponse.Asset.Video -> prepareVideoAsset(
        asset,
        loadVast.value
    )
}

private suspend fun prepareImageAsset(
    asset: NativeOrtbResponse.Asset.Image,
    mediaCacheRepository: MediaCacheRepository = SdkObjectFactory.Media.mediaCacheRepository
): Result<PreparedNativeAsset, String> =
    when (val mediaCacheResult = mediaCacheRepository.getMediaFile(asset.url)) {
        is MediaCacheRepository.Result.Success -> Result.Success(
            PreparedNativeAsset.Image(
                asset,
                mediaCacheResult.file.absolutePath
            )
        )

        else -> {
            Result.Failure(
                "media cache error: $mediaCacheResult"
            )
        }
    }

// TODO. Add/test "cancellability" internally.
// TODO. VastAdLoad.kt partial duplication.
private suspend fun prepareVideoAsset(
    asset: NativeOrtbResponse.Asset.Video,
    loadVast: VastAdLoader
): Result<PreparedNativeAsset, String> {
    return when(val vastAdResult = loadVast(adm = asset.vastTag, isStreamingEnabled = false)) {
        is Result.Success -> {
            Result.Success(PreparedNativeAsset.Video(asset, vastAdResult.value))
        }

        is Result.Failure -> {
            Result.Failure("failed to load vast ad")
        }
    }
}
