package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.view.MotionEvent
import android.webkit.*
import androidx.webkit.WebViewClientCompat
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.BaseWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.Result
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.applyCSSRenderingFix
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.StaticAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.loadDataWithDefaultBaseUrl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonRecorder
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.ScreenUtils
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import kotlin.time.Duration

private const val TAG = "StaticWebView"

private data class BannerAdTouch(
    val viewPositionX: Int,
    val viewPositionY: Int,
    val viewSizeHeight: Int,
    val viewSizeWidth: Int,
    val touchX: Int,
    val touchY: Int
)

internal typealias StaticWebViewLoadResult = Result<Unit, StaticAdError>

// TODO. Ugly. Duplication of VastWebView.
//  Review settings during webview build/init.
@SuppressLint("SetJavaScriptEnabled", "ViewConstructor")
internal class StaticWebView(
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    externalLinkHandler: ExternalLinkHandler,
) : BaseWebView(context), ButtonRecorder {

    init {
        scrollBarStyle = SCROLLBARS_INSIDE_OVERLAY
        isHorizontalScrollBarEnabled = false
        isVerticalScrollBarEnabled = false

        with(settings) {
            setSupportZoom(false)
            javaScriptEnabled = true
            domStorageEnabled = true
            allowFileAccess = false
            allowContentAccess = false
        }

        setBackgroundColor(Color.TRANSPARENT)

        visibility = GONE
    }

    private val scope = CoroutineScope(DispatcherProvider().main)

    private val webViewClientImpl = WebViewClientImpl(
        scope,
        customUserEventBuilderService,
        externalLinkHandler
    ).also {
        webViewClient = it
    }

    val unrecoverableError = webViewClientImpl.unrecoverableError
    val clickthroughEvent: SharedFlow<Unit> = webViewClientImpl.clickthroughEvent
    val isPageFinished: StateFlow<Boolean> = webViewClientImpl.isPageFinished

    override fun destroy() {
        super.destroy()
        scope.cancel()
    }

    // Touch Listener that tracks the last click event
    @SuppressLint("ClickableViewAccessibility")
    override fun onTouchEvent(event: MotionEvent): Boolean {
        if (event.action == MotionEvent.ACTION_DOWN) {
            val position = IntArray(2)
            this.getLocationOnScreen(position)
            webViewClientImpl.onLastTouch(
                BannerAdTouch(
                    viewPositionX = position.first(),
                    viewPositionY = position.last(),
                    viewSizeHeight = this.height,
                    viewSizeWidth = this.width,
                    touchX = (event.x + position.first()).toInt(),
                    touchY = (event.y + position.last()).toInt()
                )
            )
        }
        return super.onTouchEvent(event)
    }

    // TODO. Refactor.
    // Currently it's single-time use, so no worries regarding isLoaded flag volatility.
    suspend fun loadHtml(
        html: String,
        timeout: Duration
    ): StaticWebViewLoadResult = withContext(DispatcherProvider().main) {
        try {
            loadDataWithDefaultBaseUrl(applyCSSRenderingFix(html))
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "loadHtml", e)
            return@withContext Result.Failure(StaticAdError.STATIC_AD_WEBVIEW_DATA_WITH_DEFAULT_BASE_URL_ERROR)
        }

        val isLoadOperationTimedOut = withTimeoutOrNull(timeout) {
            // Awaiting for web page either loading successfully or having an error.
            webViewClientImpl.isLoaded.combine(
                webViewClientImpl.unrecoverableError
            ) { isLoaded, unrecoverableError ->
                isLoaded to unrecoverableError
            }.first {
                val (isLoaded, unrecoverableError) = it
                isLoaded || unrecoverableError != null
            }
        } == null

        // load timeout: as per https://mlc.atlassian.net/browse/SDK-1751,
        // let's set static web ad loaded state to "true".
        //
        // Once the issue https://mlc.atlassian.net/browse/SDK-1751 resolved, consider returning "timeout" StaticWebViewLoadResult
        if (isLoadOperationTimedOut) {
            webViewClientImpl.enforceIsLoaded()
        }

        // Returning latest values back for handling.
        val isLoaded = webViewClientImpl.isLoaded.value
        val unrecoverableError = webViewClientImpl.unrecoverableError.value

        if (unrecoverableError != null) {
            Result.Failure(unrecoverableError)
        } else if (isLoaded) {
            Result.Success(Unit)
        } else {
            Result.Failure(StaticAdError.STATIC_AD_UNKNOWN_ERROR)
        }
    }

    override fun onButtonRendered(button: CustomUserEventBuilderService.UserInteraction.Button) {
        webViewClientImpl.onButtonRendered(button)
    }

    override fun onButtonUnRendered(
        buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
    ) {
        webViewClientImpl.onButtonUnRendered(buttonType)
    }
}

// TODO. Refactor. Also logging for errors and stuff.
private class WebViewClientImpl(
    private val scope: CoroutineScope,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val externalLinkHandler: ExternalLinkHandler,
    private val buttonTracker: ButtonTracker = ButtonTracker(),
) : WebViewClientCompat(), ButtonRecorder {

    private val TAG = "WebViewClientImpl"

    private val _isLoaded = MutableStateFlow(false)
    val isLoaded: StateFlow<Boolean> = _isLoaded

    /**
     * Workaround: call for cases when some of the web page resources (_img_ tag for example) take too much time to load:
     * [onPageFinished] not called in a while - the place where we consider a web page/ad loaded.
     *
     * This hurts Moloco's static ad fill-rate: lots of ad load timeouts.
     *
     * Until we figure out why this happens,
     * we'll force-enable [isLoaded] flag after some predefined delay, or [onPageFinished] gets there first.
     */
    fun enforceIsLoaded() {
        _isLoaded.value = true
    }

    private val _unrecoverableError = MutableStateFlow<StaticAdError?>(null)
    val unrecoverableError = _unrecoverableError.asStateFlow()

    // TODO. Why not just a simple listener? Auto cancellation of late events is cool though.
    private val _clickthroughEvent = MutableSharedFlow<Unit>()
    val clickthroughEvent: SharedFlow<Unit> = _clickthroughEvent
    private var lastTouch: BannerAdTouch? = null

    fun onLastTouch(bannerAdTouch: BannerAdTouch) {
        lastTouch = bannerAdTouch
    }

    override fun onButtonRendered(button: CustomUserEventBuilderService.UserInteraction.Button) {
        buttonTracker.onButtonRendered(button)
    }

    override fun onButtonUnRendered(
        buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
    ) {
        buttonTracker.onButtonUnRendered(buttonType)
    }

    // Old version of the method works for api 33 as well, so I leave it as is.
    // This annotation is required by Kotlin in order to avoid warnings.
    @Deprecated("Deprecated in Android API 24")
    override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
        var modifiedUrl = url
        val eventTimestamp = System.currentTimeMillis()
        lastTouch?.let { lastTouch ->
            url?.let {
                val position = CustomUserEventBuilderService.UserInteraction.Position(
                    topLeftXDp = ScreenUtils.toDp(lastTouch.touchX),
                    topLeftYDp = ScreenUtils.toDp(lastTouch.touchY)
                )
                val clickEvent = CustomUserEventBuilderService.UserInteraction.Click(
                    clickPosition = position,
                    viewPosition = CustomUserEventBuilderService.UserInteraction.Position(
                        topLeftXDp = ScreenUtils.toDp(lastTouch.viewPositionX),
                        topLeftYDp = ScreenUtils.toDp(lastTouch.viewPositionY)
                    ),
                    viewSize = CustomUserEventBuilderService.UserInteraction.Size(
                        widthDp = ScreenUtils.toDp(lastTouch.viewSizeWidth),
                        heightDp = ScreenUtils.toDp(lastTouch.viewSizeHeight)
                    ),
                    buttonTracker.renderedButtons()
                )

                // We do a blocking coroutine invocation here to bridge between
                // old regular sequential code to have the modified url before
                // the launch of that url
                runBlocking {
                    modifiedUrl = customUserEventBuilderService
                        .userAdInteractionExtAsQueryParameter(eventTimestamp, clickEvent, url)
                }
            }
        }

        MolocoLogger.debug(TAG, "Launching url: $modifiedUrl")
        if (externalLinkHandler(modifiedUrl ?: "")) {
            scope.launch { _clickthroughEvent.emit(Unit) }
        }
        // Stop loading the url in the webview.
        return true
    }

    private val _isPageFinished = MutableStateFlow(false)
    val isPageFinished = _isPageFinished.asStateFlow()

    override fun onPageFinished(view: WebView?, url: String?) {
        super.onPageFinished(view, url)
        _isLoaded.value = true
        _isPageFinished.value = true
    }

    /**
     * Report an error to the host application. These errors are unrecoverable (i.e. the main resource is unavailable). The errorCode parameter corresponds to one of the ERROR_* constants.
     */
    // Looking for unrecoverable errors only, hence deprecated function is better for that (?)
    // This annotation is required by Kotlin in order to avoid warnings.
    @Suppress("DEPRECATION")
    @Deprecated("Deprecated in Android API 23")
    override fun onReceivedError(
        view: WebView?,
        errorCode: Int,
        description: String?,
        failingUrl: String?
    ) {
        super.onReceivedError(view, errorCode, description, failingUrl)
        _unrecoverableError.value = StaticAdError.STATIC_AD_WEBVIEW_RECEIVED_ERROR
        MolocoLogger.error(TAG, "onReceivedError $description")
    }

    override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
        // TODO. Logging.
        // https://developer.android.com/guide/webapps/managing-webview#termination-handle
        // Basically, then webview will be destroyed externally after this, which, ideally, isn't known here.
        // But who cares, plus deadlines.
        _unrecoverableError.value = StaticAdError.STATIC_AD_WEBVIEW_RENDER_PROCESS_GONE_ERROR
        MolocoLogger.error(TAG, "onRenderProcessGone")
        return true
    }
}
