package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui

import android.annotation.SuppressLint
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.requiredSize
import androidx.compose.material.Icon
import androidx.compose.material.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.semantics.testTag
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.moloco.sdk.R
import com.moloco.sdk.internal.publisher.nativead.ui.clickable
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button.ButtonType
import kotlinx.coroutines.flow.StateFlow

internal const val DEFAULT_PRIVACY_URL = "https://www.moloco.com/privacy-policy"

/**
 * A clickable AdBadge to be made visible for Admob
 *
 * This should be used take the user outside the app to our privacy page with onClick.
 *
 * @param modifier A [Modifier] for configuring the layout, behavior, and
 * appearance of this badge.
 * @param onClick A lambda function that will be called when the badge is clicked.
 */
@Composable
internal fun AdBadge(modifier: Modifier = Modifier,
                     clickUrl: String,
                     onClick: (String) -> Unit) {

    val contentDescription = "Ad Badge"
    Icon(
        modifier = modifier
            .semantics {
                contentDescription.let {
                    this.contentDescription = it
                    this.testTag = it
                }
            }
            .requiredSize(12.dp)
            .clickable { onClick(clickUrl) },
        painter = painterResource(id = R.drawable.info_badge),
        tint = Color.Unspecified,
        contentDescription = contentDescription
    )
}

typealias VastAdBadge = @Composable BoxScope.(
    onButtonRendered: (Button) -> Unit,
    currentAdViewPart: StateFlow<AdViewModel.AdPart?>?,
) -> Unit


/**
 * A trackable composable of the Ad Badge for VAST.
 * This will be rendered at different stages of the VAST ad. Currently we only track at
 * Video playback and at Companion. DEC is pending.
 */
@SuppressLint("ComposableNaming")
@Composable
internal fun defaultVastAdBadge(
    modifier: Modifier = Modifier,
    clickUrl: String = DEFAULT_PRIVACY_URL,
    externalLinkHandler: ExternalLinkHandler = SdkObjectFactory
        .Miscellaneous.externalLinkHandlerFactory,
    onClick: (String) -> Unit = { externalLinkHandler(clickUrl) }
): VastAdBadge = { onButtonRendered, currentAdPartFlow  ->
    currentAdPartFlow?.let { flow ->
        val currentAdPart by flow.collectAsState()
        when (currentAdPart) {
            is AdViewModel.AdPart.Linear -> {
                TrackableButton(
                    buttonType = ButtonType.AD_BADGE,
                    onButtonRendered = onButtonRendered
                ) { trackableModifier ->
                    AdBadge(
                        modifier = modifier.then(trackableModifier),
                        clickUrl,
                        onClick
                    )
                }
            }
            is AdViewModel.AdPart.Companion -> {
                TrackableButton(
                    buttonType = ButtonType.AD_BADGE,
                    onButtonRendered = onButtonRendered
                ) { trackableModifier ->
                    AdBadge(
                        modifier = modifier.then(trackableModifier),
                        clickUrl,
                        onClick
                    )
                }
            }
            is AdViewModel.AdPart.DEC -> {/* NO OP */ }
            is AdViewModel.AdPart.Mraid -> {/* NO OP */ }
            null -> {/* NO OP */ }
        }
    }
}


typealias WebViewAdBadge = @Composable BoxScope.(
    onButtonRendered: (Button) -> Unit
) -> Unit

/**
 * A trackable composable for WebView based render scenarios - this includes Banners, MRAID, Static.
 */
@SuppressLint("ComposableNaming")
@Composable
internal fun defaultWebViewAdBadge(
    modifier: Modifier = Modifier,
    clickUrl: String = DEFAULT_PRIVACY_URL,
    externalLinkHandler: ExternalLinkHandler = SdkObjectFactory
        .Miscellaneous.externalLinkHandlerFactory,
    onClick: (String) -> Unit = { externalLinkHandler(clickUrl) }
): WebViewAdBadge = { onButtonRendered  ->
    TrackableButton(
        buttonType = ButtonType.AD_BADGE,
        onButtonRendered = onButtonRendered
    ) { trackableModifier ->
        AdBadge(
            modifier = modifier.then(trackableModifier),
            clickUrl,
            onClick
        )
    }
}

@Preview(showBackground = true)
@Composable
private fun AdBadgePreview() {
    MaterialTheme {
        Box(modifier = Modifier.background(Color.White)) {
            AdBadge(Modifier, "", onClick = {})
        }
    }
}
