package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui

import androidx.compose.foundation.layout.BoxScope
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.GoNextAction
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.StateFlow

// https://mlc.atlassian.net/browse/SDK-1084
// Main place for skip/close, that is, "next ad part" type buttons of VastRender;
// for linears, companions, DECs etc.
@Composable
internal fun BoxScope.AdGoNextButton(
    adViewModel: AdViewModel,
    currentAdPart: AdViewModel.AdPart?,
    adSkipCountdownButton: AdCountdownButton?,
    adCloseCountdownButton: AdCountdownButton?
) {
    val countdownButton = when (currentAdPart?.isLastAdPart) {
        true -> adCloseCountdownButton
        false -> adSkipCountdownButton
        null -> null
    }

    countdownButton?.let {
        val goNextButton = @Composable {
            GoNextButton(
                adViewModel.goNextAction,
                adViewModel::goNextAdPartOrDismissAd,
                adViewModel::onButtonRendered,
                countdownButton
            )
        }

        // https://mlc.atlassian.net/browse/SDK-1308
        // Fix for Telepathy tracking not registering new buttons rendering over each new ad part.
        // Similar fix to what we already have in MolocoVastCTA.kt
        // (failed to find JIRA task).
        when (currentAdPart) {
            is AdViewModel.AdPart.Companion -> goNextButton()
            is AdViewModel.AdPart.DEC -> goNextButton()
            is AdViewModel.AdPart.Linear -> goNextButton()
            else -> { /* no-op */ }
        }
    }
}

@Composable
private fun BoxScope.GoNextButton(
    goNextAction: StateFlow<GoNextAction.State>,
    onGoNext: () -> Unit,
    onButtonRendered: (CustomUserEventBuilderService.UserInteraction.Button) -> Unit,
    countdownButton: AdCountdownButton
) {
    val goNextActionState by goNextAction.collectAsStateWithLifecycle()

    countdownButton(
        // secondsLeft
        (goNextActionState as? GoNextAction.State.Countdown)?.seconds?.toInt() ?: 0,
        // enabled
        goNextActionState is GoNextAction.State.Available,
        // visible
        goNextActionState !is GoNextAction.State.Unavailable,
        // onClick
        onGoNext,
        onButtonRendered
    )
}
