package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.Intent.FLAG_ACTIVITY_NEW_TASK
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.BackHandler
import androidx.activity.compose.setContent
import androidx.compose.runtime.Composable
import androidx.compose.ui.viewinterop.AndroidView
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.autoStoreOnComplete
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.autoStoreOnSkip
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.closeDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.decDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.overrideSkipEnabled
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.overrideSkipEnabledDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putAutoStoreOnComplete
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putAutoStoreOnSkip
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putCloseDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putDECDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putOverrideSkipEnabled
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putOverrideSkipEnabledDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putStartMuted
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.startMuted
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.HideSystemUI
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Ad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.vastAdPlaylistController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.VastRenderer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultVastRenderer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.lang.ref.WeakReference

// TODO. Remove weak references. There's no need.
internal class VastActivity : ComponentActivity() {
    val customUserEventBuilderService by lazy { SdkObjectFactory.UserTracking.customUserEventBuilderFactory }

    private var ac: AdController? = null

    private val scope = CoroutineScope(DispatcherProvider().main)

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val elh = SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory
        val activity = this

        val ad = ad
        if (ad == null) {
            MolocoLogger.info("VastActivity", "ad is missing")
            finish()
            return
        }

        val vastRenderer = VastRenderer
        if (vastRenderer == null) {
            MolocoLogger.info("VastActivity", "VastRenderer is missing")
            finish()
            return
        }

        val adController = vastAdPlaylistController(
            ad,
            elh,
            activity,
            customUserEventBuilderService,
            mute = intent.startMuted,
            overrideLinearGoNextActionEnabled = intent.overrideSkipEnabled,
            overrideLinearGoNextActionEnabledDelaySeconds = intent.overrideSkipEnabledDelaySeconds,
            companionGoNextActionDelaySeconds = intent.closeDelaySeconds,
            decGoNextActionDelaySeconds = intent.decDelaySeconds,
            autoStoreOnSkip = intent.autoStoreOnSkip,
            autoStoreOnComplete = intent.autoStoreOnComplete
        )

        ac = adController

        updateCurrentVastActivity(this)

        // Subscribing to ad events FIRST and delegating them to the singleton CurrentVastAdEvents.
        adController.event.onEach {
            CurrentVastAdEvents.emit(it)
            // TODO. Duplicate.
            if (it.isErrorOrDismiss) {
                finish()
            }
        }.launchIn(scope)

        setContent {
            VastView(adController, vastRenderer, watermark)
        }

        adController.show()
    }

    override fun onDestroy() {
        super.onDestroy()

        onAdClose?.invoke()
        ac?.destroy()
        ac = null

        scope.cancel()
        updateCurrentVastActivity(null)
    }

    // TODO - Companion object is keeping global state that requires careful state
    // updates whenever activity lifecycle changes (onCreate, onDestroy).
    // Refactor this so that this state is held by a scoped service/object
    // that is injected into the activity
    // which is updated during activity lifecycle callbacks.
    companion object {
        private val CurrentVastAdEvents = MutableSharedFlow<AdControllerEvent>()

        private fun updateCurrentVastActivity(activity: VastActivity?) {
            weakCurrentVastActivity = WeakReference(activity)
            if (activity == null) {
                cancelEventListener()
            }
        }

        private fun getCurrentVastActivity(): VastActivity? = weakCurrentVastActivity.get()

        private var weakCurrentVastActivity = WeakReference<VastActivity>(null)

        private var ad: Ad? = null
        private var VastRenderer: VastRenderer? = null

        private var onAdClose: (() -> Unit)? = null
        private var eventListenerJob: Job? = null
        private var watermark: Watermark? = null

        // TODO. Rename? Refactor? Cold flow?
        // Note: activity gets closed automatically if the scope is cancelled
        // or ad is dismissed/errored.
        suspend fun showAd(
            ad: Ad,
            context: Context,
            options: VastOptions,
            onAdClose: () -> Unit,
            watermark: Watermark,
            onAdEvent: (event: AdControllerEvent) -> Unit,
        ) {
            this.watermark = watermark
            this.onAdClose = onAdClose
            withContext(DispatcherProvider().main) {
                try {
                    Companion.ad = ad
                    VastRenderer = options.VastRenderer

                    coroutineScope {
                        // Since the context is Main, this job will be executed first...
                        // Event subscription.
                        eventListenerJob = launch {
                            CurrentVastAdEvents
                                .onEach { onAdEvent(it) }
                                // TODO. Duplicate.
                                .firstOrNull { it.isErrorOrDismiss }
                        }
                        // ... and this one will be the last.
                        // Once we are subscribed for events, let's show the ad activity.
                        launch {
                            context.startActivity(
                                Intent(context, VastActivity::class.java).apply {
                                    putStartMuted(options.startMuted)
                                    putCloseDelaySeconds(options.companionGoNextActionDelaySeconds)
                                    putDECDelaySeconds(options.decGoNextActionDelaySeconds)
                                    putOverrideSkipEnabled(
                                        options.overrideLinearGoNextActionEnabled
                                    )
                                    putOverrideSkipEnabledDelaySeconds(
                                        options.overrideLinearGoNextActionEnabledDelaySeconds
                                    )
                                    putAutoStoreOnSkip(options.autoStoreOnSkip)
                                    putAutoStoreOnComplete(options.autoStoreOnComplete)

                                    flags = FLAG_ACTIVITY_NEW_TASK
                                }
                            )
                        }
                    }
                } finally {
                    // If the parent scope is cancelled or error/dismiss event,
                    // destroy the current activity to avoid leaks.
                    getCurrentVastActivity()?.finish()
                    Companion.onAdClose = null
                    Companion.ad = null
                    VastRenderer = null

                    // Stop listening for the vast ad events
                    cancelEventListener()
                }
            }
        }

        private fun cancelEventListener() {
            if (eventListenerJob?.isActive == true) {
                eventListenerJob?.cancel()
                eventListenerJob = null
            }
        }

        private val AdControllerEvent.isErrorOrDismiss: Boolean
            get() = this is AdControllerEvent.Error || this == AdControllerEvent.Dismiss
    }
}

@Composable
private fun Activity.VastView(
    ac: AdController,
    vastRenderer: VastRenderer = defaultVastRenderer(),
    watermark: Watermark?
) {

    AndroidView(
        factory = { ctx -> vastRenderer(ctx, ac) }
    ).let { view -> watermark?.ApplyWatermark { view } ?: view }

    BackHandler(onBack = ac::goNextAdPartOrDismissAd)
    HideSystemUI()
}
