package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast

import android.content.Context
import com.moloco.sdk.common_adapter_internal.ScreenData
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.ConnectivityService
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.ScreenService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.Result
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdLoadError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaConfig
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.MediaStreamStatus
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.AdChild
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Companion
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Creative
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.CreativeChild
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Icon
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Impression
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.InLine
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Linear
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.MediaFile
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Tracking
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.TrackingEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Vast
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VastError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VideoClick
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VideoClicks
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Wrapper
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Ad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.LinearProgressTracking
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.LinearTracking
import com.moloco.sdk.xenoss.sdkdevkit.android.core.requestTimeoutMillis
import io.ktor.client.HttpClient
import io.ktor.client.plugins.HttpRequestTimeoutException
import io.ktor.client.request.get
import io.ktor.client.statement.bodyAsText
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.cancellable
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration

private typealias RenderAd = Ad
private typealias RenderAdResult = Result<RenderAd, VastAdLoadError>
private typealias RenderLinear = com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Linear
private typealias RenderLinearResult = Result<RenderLinear, VastAdLoadError>
private typealias RenderCompanion = com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Companion
private typealias RenderIcon = com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Icon

internal fun VastAdLoader(context: Context): VastAdLoader = VastAdLoaderImpl(
    VastParser(),
    SdkObjectFactory.Media.mediaConfigSingleton,
    SdkObjectFactory.Media.mediaCacheRepository,
    VastTracker(),
    SdkObjectFactory.DeviceAndApplicationInfo.connectivityService,
    SdkObjectFactory.Network.httpClientSingleton,
    ScreenService(context)
)

typealias VastAdLoadResult = Result<RenderAd, VastAdLoadError>

// TODO. Refactor - This class has a lot of business logic, need to refactor to better test it
// What a nice name eh?
// I expect this interface to take care of:
// tag/media filtering/aggregation and error notification + media file sorting + linear ad precaching.
// As an end result I expect all required data for Vast player to be collected.
internal interface VastAdLoader {
    suspend operator fun invoke(adm: String, mtid: String = "UNKNOWN_MTID", isStreamingEnabled: Boolean): VastAdLoadResult

    /**
     * Waits for the media file to start streaming until the timeout and returns the result.
     * If the media file has not fetched enough during the timeout, then it returns an error.
     */
    suspend fun waitForAdLoadToStart(ad: RenderAd, timeout: Duration): VastAdLoadResult
}

private class VastAdLoaderImpl(
    private val parseVast: VastParser,
    private val mediaConfig: MediaConfig,
    private val mediaCacheRepository: MediaCacheRepository,
    private val vastTracker: VastTracker,
    private val connectivityService: ConnectivityService,
    private val httpClient: HttpClient,
    private val screenService: ScreenService
) : VastAdLoader {
    private val TAG = "VastAdLoaderImpl"

    override suspend fun waitForAdLoadToStart(ad: RenderAd, timeout: Duration): VastAdLoadResult {
        // 1. Wait for the media file to start streaming
        MolocoLogger.info(TAG, "Waiting for $timeout to load the vast media file: $mediaCacheRepository")
        val streamStatus = withTimeoutOrNull(timeout) {
            mediaCacheRepository.streamMediaFileStatus(ad.linear.networkMediaResource)
                .firstOrNull {

                    if (it is MediaStreamStatus.InProgress) {
                        MolocoLogger.info(TAG, "Stream status: ${it.progress.bytesDownloaded}/${it.progress.totalBytes} bytes downloaded")
                    }
                    it is MediaStreamStatus.Complete || it is MediaStreamStatus.Failure
                }
        }
        MolocoLogger.info(TAG, "Either timeout occurred or media file streaming had terminal status")

        MolocoLogger.info(TAG, "Stream status: $streamStatus on timeout")
        // 2. Media file should be > 0 bytes and should exist, and should have enough playable content
        if (streamStatus == null) {
            // If stream status is null, then it means that the media cache repository was still fetching the
            // media
            val adFile = ad.linear.localMediaResource
            if (!adFile.exists() || adFile.length() == 0L) {
                MolocoLogger.error(TAG, "${adFile.absolutePath} does not exist or is empty")
                MolocoLogger.error(TAG, "Failed to start streaming media file, reporting timeout error")
                return Result.Failure(VastAdLoadError.VAST_AD_LOAD_MEDIA_FILE_TIMEOUT_ERROR)
            }
            MolocoLogger.info(TAG, "Local vast media resource exists and has some content. Checking for bitrate information")
            if (ad.linear.localMediaResourceBitrate != null) {
                MolocoLogger.info(TAG, "Checking for playability of VAST ad with bitrate: ${ad.linear.localMediaResourceBitrate}")
                val duration = calculateDurationSecs(adFile.length(), ad.linear.localMediaResourceBitrate)
                MolocoLogger.info(TAG, "VAST ad has playable duration: $duration seconds")
                if (duration < mediaConfig.minStreamingPlayableDurationOnTimeoutSecs) {
                    MolocoLogger.error(TAG, "VAST does not have enough playable duration, so failing ")
                    return Result.Failure(VastAdLoadError.VAST_AD_LOAD_MEDIA_FILE_TIMEOUT_NOT_ENOUGH_PLAYABLE_ERROR)
                }
            } else {
                MolocoLogger.info(TAG, "VAST ad playable duration cannot be determined due to no bitrate information")
                return Result.Failure(VastAdLoadError.VAST_AD_LOAD_MEDIA_FILE_UNABLE_TO_DETERMINE_TIMEOUT_ERROR)
            }
        } else {
            if (streamStatus is MediaStreamStatus.Complete) {
                MolocoLogger.info(TAG, "Streamed entire file successfully")
                return Result.Success(ad)
            } else if (streamStatus is MediaStreamStatus.Failure) {
                MolocoLogger.info(TAG, "Failed to stream file")
                return Result.Failure(mapMediaErrorToVastError(streamStatus.failure))
            }
        }

        MolocoLogger.info(TAG, "Media file partially exists and ready for streaming")
        return Result.Success(ad)
    }

    private fun calculateDurationSecs(fileSizeInBytes: Long, bitrateInKbps: Int): Double {
        val fileSizeInBits = fileSizeInBytes * 8 // Convert bytes to bits
        val bitrateInBps = bitrateInKbps * 1000 // Convert kbps to bps
        return fileSizeInBits.toDouble() / bitrateInBps // Calculate duration in seconds
    }

    override suspend operator fun invoke(adm: String, mtid: String, isStreamingEnabled: Boolean): VastAdLoadResult {
        val vast = when (val parsedResult = parseVast(adm)) {
            is Result.Failure -> {
                MolocoLogger.error(TAG, "Failed to parse vast XML: ${parsedResult.value}")
                return Result.Failure(parsedResult.value)
            }

            is Result.Success -> {
                parsedResult.value
            }
        }

        val renderAdResult = withContext(DispatcherProvider().default) {
            tryLoadVastRenderAd(
                vast,
                wrapperChainParams = null,
                calculateTargetLinearFileSizeInMegabytes(),
                screenService(),
                isStreamingEnabled,
                mtid,
            )
        }
        return when(renderAdResult) {
            is Result.Failure -> Result.Failure(renderAdResult.value)
            is Result.Success -> Result.Success(renderAdResult.value)
        }
    }

    // MoPub: Video bitrate combined with video duration should net out
    // to a recommended file size of 2MB (recommended maximum 10MB).
    private fun calculateTargetLinearFileSizeInMegabytes() =
        if (!connectivityService.isNetworkMetered()) {
            10.0
        } else {
            2.0
        }

    private data class WrapperChainParams(
        val wrapperDepth: Int,
        val usedVastAdTagUrls: Set<String>,
        val followAdditionalWrappers: Boolean,
        val aggregatedWrapperChainData: AggregatedWrapperChainAdData
    )

    private data class AggregatedWrapperChainAdData(
        val impressions: List<Impression>,
        val errorUrls: List<String>,
        // TODO. Naming is retarded.
        // The last item is from the deepest Wrapper.
        val creativesPerWrapper: List<List<Creative>>,
    )

    private fun sendVastError(errorUrls: List<String>, error: VastError? = null) {
        vastTracker.track(errorUrls, error)
    }

    private suspend fun tryLoadVastRenderAd(
        vast: Vast,
        wrapperChainParams: WrapperChainParams? = null,
        targetLinearFileSizeInMegabytes: Double,
        screenData: ScreenData,
        isStreamingEnabled: Boolean,
        mtid: String,
    ): RenderAdResult {
        MolocoLogger.info(TAG, "Loading vast ad with wrapperChainParams: $wrapperChainParams")
        val aggregatedErrorUrls =
            wrapperChainParams?.aggregatedWrapperChainData?.errorUrls + vast.errorUrl

        // "No ads" error notification for Wrapper/Inline Vast response.
        if (vast.ads.isEmpty()) {
            sendVastError(
                aggregatedErrorUrls,
                wrapperChainParams?.let { VastError.WrapperNoAds }
            )
            return Result.Failure(VastAdLoadError.VAST_AD_LOAD_NO_ADS_ERROR)
        }

        // Though not required, let's put the root Vast error tag into the wrapper chain. Just in case..
        val updatedWrapperChainParams = wrapperChainParams?.copy(
            aggregatedWrapperChainData = wrapperChainParams.aggregatedWrapperChainData.copy(
                errorUrls = aggregatedErrorUrls
            )
        )

        var adLoadError: VastAdLoadError = VastAdLoadError.VAST_AD_LOAD_RENDER_AD_LOAD_ERROR
        val adRender = vast.ads
            // Following MoPub's approach:
            // https://developers.mopub.com/dsps/ad-formats/video/#additional-vast-30-behaviors
            .filter { it.sequence == null || it.sequence in 0..1 }
            .sortedBy { it.sequence }
            .asFlow()
            .cancellable()
            .mapNotNull { ad ->
                val result = when (val child = ad.child) {
                    is AdChild.Wrapper -> {
                        MolocoLogger.info(TAG, "Found Wrapper child element, trying load wrapper render Ad")
                        tryLoadWrapperRenderAd(
                            child.wrapper,
                            updatedWrapperChainParams,
                            targetLinearFileSizeInMegabytes,
                            screenData,
                            isStreamingEnabled,
                            mtid,
                        )
                    }
                    is AdChild.InLine -> {
                        MolocoLogger.info(TAG, "Found InLine child element, trying load render Ad")
                        tryLoadInLineRenderAd(
                            child.inline,
                            updatedWrapperChainParams?.aggregatedWrapperChainData,
                            targetLinearFileSizeInMegabytes,
                            screenData,
                            isStreamingEnabled,
                            mtid,
                        )
                    }
                }

                when(result) {
                    is Result.Failure -> {
                        MolocoLogger.error(TAG, "Failed to load the ad with error: ${result.value}")
                        // Store the last error for the last child. We need to send an error in case no child could be loaded.
                        adLoadError = result.value
                        // return null on error so we try other childs
                        null
                    }
                    is Result.Success -> result.value
                }
            }
            .firstOrNull() ?: return Result.Failure<RenderAd, VastAdLoadError>(adLoadError).also {
            MolocoLogger.error(TAG, "Failed to load linear: $adLoadError")
        }

        return Result.Success(adRender)
    }

    private suspend fun tryLoadWrapperRenderAd(
        wrapper: Wrapper,
        wrapperChainParams: WrapperChainParams?,
        targetLinearFileSizeInMegabytes: Double,
        screenData: ScreenData,
        isStreamingEnabled: Boolean,
        mtid: String,
    ): RenderAdResult {
        MolocoLogger.info(TAG, "Loading wrapper vast ad: ${wrapper.vastAdTagUrl}")
        val currentWrapperDepth = wrapperChainParams?.wrapperDepth?.inc() ?: 0

        val aggregatedErrorUrls =
            wrapperChainParams?.aggregatedWrapperChainData?.errorUrls + wrapper.errorUrls

        if (currentWrapperDepth > WRAPPER_DEPTH_MAX ||
            wrapperChainParams?.usedVastAdTagUrls?.contains(wrapper.vastAdTagUrl) == true ||
            wrapperChainParams?.followAdditionalWrappers == false
        ) {
            sendVastError(aggregatedErrorUrls, VastError.WrapperLimit)
            val error = VastAdLoadError.VAST_AD_LOAD_WRAPPER_LIMIT_ERROR
            MolocoLogger.error(TAG, "Failed to load wrapper vast ad: $error")
            return Result.Failure(error)
        }

        val vast = when(val vastParserResult = loadAndParseWrapperVastDocument(wrapper, aggregatedErrorUrls)) {
            is Result.Failure -> {
                MolocoLogger.error(TAG, "Failed to load wrapper vast ad: ${vastParserResult.value}")
                return Result.Failure(vastParserResult.value)
            }
            is Result.Success -> vastParserResult.value
        }

        val aggregatedImpressions =
            wrapperChainParams?.aggregatedWrapperChainData?.impressions + wrapper.impressions

        val creativesPerWrapper =
            wrapperChainParams?.aggregatedWrapperChainData?.creativesPerWrapper + listOf(wrapper.creatives)

        val usedVastAdTagUrls =
            wrapperChainParams?.usedVastAdTagUrls + wrapper.vastAdTagUrl

        return tryLoadVastRenderAd(
            vast,
            WrapperChainParams(
                currentWrapperDepth,
                usedVastAdTagUrls,
                wrapper.followAdditionalWrappers ?: true,
                AggregatedWrapperChainAdData(
                    aggregatedImpressions,
                    aggregatedErrorUrls,
                    creativesPerWrapper
                )
            ),
            targetLinearFileSizeInMegabytes,
            screenData,
            isStreamingEnabled,
            mtid,
        )
    }

    private suspend fun loadAndParseWrapperVastDocument(
        wrapper: Wrapper,
        vastErrorUrls: List<String>
    ): VastParserResult {
        val rawVastXml = try {
            // TODO. IO dispatcher?
            MolocoLogger.info(TAG, "Fetching wrapper vast tag url: ${wrapper.vastAdTagUrl}")
            httpClient.get(wrapper.vastAdTagUrl) {
                requestTimeoutMillis(WRAPPER_TIMEOUT_MILLIS)
            }.bodyAsText()
        } catch (e: HttpRequestTimeoutException) {
            sendVastError(vastErrorUrls, VastError.WrapperTimeout)
            MolocoLogger.error(TAG, "Fetching wrapper vast tag url timed out", e)
            return Result.Failure(VastAdLoadError.VAST_AD_LOAD_WRAPPER_TIMEOUT_ERROR)
        } catch (e: Exception) {
            sendVastError(vastErrorUrls, VastError.Wrapper)
            MolocoLogger.error(TAG, "Fetching wrapper vast tag url fetch error", e)
            return Result.Failure(VastAdLoadError.VAST_AD_LOAD_WRAPPER_FETCH_ERROR)
        }

        val vast = (parseVast(rawVastXml) as? Result.Success)?.value
        if (vast == null) {
            sendVastError(vastErrorUrls, VastError.XmlParsing)
            MolocoLogger.error(TAG, "Failed to create VAST object from XML")
            return Result.Failure(VastAdLoadError.VAST_AD_LOAD_XML_PARSE_ERROR)
        }
        return Result.Success(vast)
    }

    // TODO. Refactor. Gigantic POS.
    private suspend fun tryLoadInLineRenderAd(
        inline: InLine,
        aggregatedWrapperChainData: AggregatedWrapperChainAdData?,
        targetLinearFileSizeInMegabytes: Double,
        screenData: ScreenData,
        isStreamingEnabled: Boolean,
        mtid: String,
    ): RenderAdResult {
        MolocoLogger.info(TAG, "Trying to load RenderAd")
        val aggregatedErrorUrls = aggregatedWrapperChainData?.errorUrls + inline.errorUrls

        if (inline.creatives.isEmpty()) {
            // TODO. Correct error?
            sendVastError(aggregatedErrorUrls, VastError.Linear)
            MolocoLogger.error(TAG, "No creatives in InLine")
            return Result.Failure(VastAdLoadError.VAST_AD_LOAD_INLINE_CREATIVES_EMPTY_ERROR)
        }

        val preparedWrapperDataForInline by lazy {
            prepareWrapperDataForInline(aggregatedWrapperChainData)
        }

        var renderLinearNullable: RenderLinear? = null
        var renderCompanion: RenderCompanion? = null

        // When we made this change, we would loop over all the creatives
        // and if none of the linear could be loaded, we'd return null to indicate failure
        // However, we need to send an error in this case, so we need to keep track of the last error
        var renderLinearError: VastAdLoadError =
            VastAdLoadError.VAST_AD_LOAD_INLINE_CREATIVES_NO_LINEAR_ERROR

        for (creative in inline.creatives) {
            if (renderLinearNullable != null && renderCompanion != null) {
                break
            }

            if (creative.hasApiFramework) {
                continue
            }

            if (renderLinearNullable == null && creative.child is CreativeChild.Linear) {
                val renderLinearResult = tryPrepareInLineRenderLinear(
                    creative.child.linear,
                    preparedWrapperDataForInline.linearTrackingList,
                    preparedWrapperDataForInline.videoClicks,
                    aggregatedErrorUrls,
                    targetLinearFileSizeInMegabytes,
                    creative.child.linear.durationMillis,
                    screenData,
                    isStreamingEnabled,
                    mtid,
                )

                when(renderLinearResult) {
                    is Result.Failure -> {
                        MolocoLogger.error(TAG, "Failed to prepare RenderLinear: ${renderLinearResult.value}")
                        renderLinearError = renderLinearResult.value
                    }
                    is Result.Success -> {
                        renderLinearNullable = renderLinearResult.value
                    }
                }
            }

            if (renderCompanion == null && creative.child is CreativeChild.Companions) {
                renderCompanion = tryPrepareRenderCompanion(creative.child.companions, screenData)
            }
        }

        // There's no point to render the ad without linear.
        var renderLinear = if (renderLinearNullable == null) {
            sendVastError(aggregatedErrorUrls, VastError.LinearFileNotFound)
            MolocoLogger.error(TAG, "Failed to load linear: $renderLinearError")
            return Result.Failure(renderLinearError)
        } else {
            MolocoLogger.info(TAG, "RenderAd loaded successfully.")
            renderLinearNullable
        }

        // If inline wasn't able to provide an icon, let's try again but with Wrappers.
        if (renderLinear.icon == null) {
            val icon =
                preparedWrapperDataForInline.iconsPerWrapper
                    // The last wrapper is the deepest/closest to the InLine, hence, .asReversed() is used.
                    .asReversed()
                    .asSequence()
                    .mapNotNull { tryPrepareRenderIcon(it) }
                    .firstOrNull()

            renderLinear = renderLinear.copy(icon = icon)
        }

        // If inline wasn't able to provide a companion ad, let's try again but with Wrappers.
        if (renderCompanion == null) {
            renderCompanion =
                preparedWrapperDataForInline.companionsPerWrapper
                    // The last wrapper is the deepest/closest to the InLine, hence, .asReversed() is used.
                    .asReversed()
                    .asSequence()
                    .mapNotNull { tryPrepareRenderCompanion(it, screenData) }
                    .firstOrNull()
        }

        // TODO. Duplication.
        val aggregatedImpressionUrls =
            (aggregatedWrapperChainData?.impressions + inline.impressions)
                .map { it.impressionUrl }

        MolocoLogger.info(TAG, "Returning RenderAd")
        return Result.Success(com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.RenderAd(
            renderLinear,
            renderCompanion,
            aggregatedImpressionUrls,
            aggregatedErrorUrls
        ))
    }

    //  TODO. Portrait/Landscape creative groups?
    private suspend fun tryPrepareInLineRenderLinear(
        linear: Linear,
        wrapperLinearTrackingList: List<Tracking>?,
        wrapperVideoClicks: VideoClicks?,
        aggregatedErrorUrls: List<String>,
        targetLinearFileSizeInMegabytes: Double,
        adDurationMillis: Long?,
        screenData: ScreenData,
        isStreamingEnabled: Boolean,
        mtid: String,
    ): RenderLinearResult {
        MolocoLogger.info(TAG, "Preparing InLine RenderLinear with target linear size: $targetLinearFileSizeInMegabytes")
        val supportedMediaFiles =
            linear.mediaFiles
                .filter {
                    !it.hasApiFramework && it.isProgressiveDelivery && isCompatibleLinearType(it.type)
                }
                .sortedWith(
                    getMediaFileComparatorBestFirst(
                        targetLinearFileSizeInMegabytes,
                        adDurationMillis,
                        screenData.widthPx,
                        screenData.heightPx
                    )
                )

        if (supportedMediaFiles.isEmpty()) {
            sendVastError(aggregatedErrorUrls, VastError.LinearNotSupportedMedia)
            return Result.Failure(VastAdLoadError.VAST_AD_LOAD_LINEAR_NOT_SUPPORTED_MEDIA_ERROR)
        }

        var vastError: VastAdLoadError = VastAdLoadError.VAST_AD_LOAD_MEDIA_FILE_UNKNOWN_ERROR

        val precachedMediaFileData =
            supportedMediaFiles
                .asFlow()
                .cancellable()
                .mapNotNull { vastMediaFile ->
                    if (isStreamingEnabled) {
                        val result =
                            mediaCacheRepository.streamMediaFile(vastMediaFile.mediaFileUrl, mtid)
                        when(result) {
                            is MediaStreamStatus.Complete -> {
                                vastMediaFile to result.file
                            }
                            is MediaStreamStatus.InProgress -> {
                                vastMediaFile to result.file
                            }
                            is MediaStreamStatus.Failure -> {
                                vastError = mapMediaErrorToVastError(result.failure)
                                null
                            }
                        }

                    } else {
                        // If we find a file, we can stop the flow (firstOrNull),
                        // but if we encounter an error, we need to continue to the next media file. So we store the error of our last media file attempt
                        when (val result =
                            mediaCacheRepository.getMediaFile(vastMediaFile.mediaFileUrl)) {
                            is MediaCacheRepository.Result.Success -> {
                                vastMediaFile to result.file
                            }

                            is MediaCacheRepository.Result.Failure -> {
                                vastError = mapMediaErrorToVastError(result)
                                null
                            }
                        }
                    }
                }
                .firstOrNull() ?: return Result.Failure<RenderLinear, VastAdLoadError>(vastError).also {
                MolocoLogger.error(TAG, "Failed to load media file: $vastError")
            }


        val (vastMediaFile, localMediaResourceFile) = precachedMediaFileData
        MolocoLogger.info(TAG, "Found a RenderLinear MediaFile: ${localMediaResourceFile.absolutePath} for url: ${vastMediaFile.mediaFileUrl}")
        // Wrapper isn't allowed to contain clickthrough url.
        // TODO. So.. you know. Why wouldn't you create a standalone WrapperCreative
        //  instead of using the same Creative class for Wrapper and Inline?
        val clickThroughUrl = linear.videoClicks?.clickThrough?.url

        /**
         * Combined Inline and Wrapper TrackingList
         */
        val aggregatedLinearTrackingList = linear.trackingList + wrapperLinearTrackingList

        /**
         * Combined Inline, Custom and Wrapper clicks
         */
        val aggregatedVideoClicks = VideoClicks(
            // TODO. Refactor. Not needed here.
            linear.videoClicks?.clickThrough,
            linear.videoClicks?.clickTrackingList + wrapperVideoClicks?.clickTrackingList,
            linear.videoClicks?.customClickList + wrapperVideoClicks?.customClickList
        )

        MolocoLogger.info(TAG, "Returning RenderLinear for url: ${vastMediaFile.mediaFileUrl}, with bitrate: ${vastMediaFile.bitrate} ")
        return Result.Success(com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.RenderLinear(
            linear.skipOffset,
            localMediaResourceFile,
            vastMediaFile.bitrate,
            vastMediaFile.mediaFileUrl,
            clickThroughUrl,
            createLinearTracking(aggregatedLinearTrackingList, aggregatedVideoClicks),
            tryPrepareRenderIcon(linear.icons)
        ))
    }

    // TODO. Refactor. Naming is retarded.
    // Basically, mapping of the wrapper chain data suitable for InLine loading/aggregation/filtering code.
    private fun prepareWrapperDataForInline(
        aggregatedWrapperChainData: AggregatedWrapperChainAdData?
    ): PreparedWrapperDataForInline {
        val linearTrackingList = mutableListOf<Tracking>()
        val videoClickTrackingList = mutableListOf<VideoClick>()
        val customVideoClickTrackingList = mutableListOf<VideoClick>()
        // TODO. Naming is retarded.
        val iconsPerWrapper = mutableListOf<List<Icon>>()
        val companionsPerWrapper = mutableListOf<List<Companion>>()

        aggregatedWrapperChainData
            ?.creativesPerWrapper
            ?.forEach { wrapperCreatives ->

                // TODO. Naming is retarded.
                val iconsPerSingleWrapper = mutableListOf<Icon>()
                val companionsPerSingleWrapper = mutableListOf<Companion>()

                wrapperCreatives
                    // TODO. Remove framework filtering? Not sure about this one for wrappers..
                    .filter { !it.hasApiFramework }
                    .forEach { wrapperCreative ->
                        when (wrapperCreative.child) {
                            // TODO. Wrapper's Linear doesn't contain mediafiles and clickthrough.
                            is CreativeChild.Linear -> {
                                val linear = wrapperCreative.child.linear

                                linearTrackingList += linear.trackingList

                                linear.videoClicks?.let {
                                    videoClickTrackingList += it.clickTrackingList
                                    customVideoClickTrackingList += it.customClickList
                                }

                                // Gathering all the icons from all the linears within a single wrapper.
                                // I care only about the wrapper icon being the closest to the InLine.
                                // So putting all the icons in the same group is fine, since we display only a single icon.
                                iconsPerSingleWrapper += linear.icons
                            }
                            is CreativeChild.Companions -> {
                                // Collecting all the companions from all the companions within a single wrapper.
                                // I care only about the wrapper companion being the closest to the InLine.
                                // So putting all the companions in the same group is fine, since we display only a single companion.
                                companionsPerSingleWrapper += wrapperCreative.child.companions
                            }
                        }
                    }

                // TODO. Naming is retarded.
                iconsPerWrapper += iconsPerSingleWrapper
                companionsPerWrapper += companionsPerSingleWrapper
            }

        return PreparedWrapperDataForInline(
            linearTrackingList,
            VideoClicks(null, videoClickTrackingList, customVideoClickTrackingList),
            iconsPerWrapper,
            companionsPerWrapper
        )
    }

    // TODO. Naming is retarded.
    private class PreparedWrapperDataForInline(
        val linearTrackingList: List<Tracking>,
        val videoClicks: VideoClicks?,
        val iconsPerWrapper: List<List<Icon>>,
        val companionsPerWrapper: List<List<Companion>>
    )

    //  TODO. Portrait/Landscape creative groups?
    //   Also check for supported files in <StaticResource>: .png, .bmp, .jpeg, and .gif
    private fun tryPrepareRenderCompanion(
        companions: List<Companion>,
        screenData: ScreenData
    ): RenderCompanion? = companions
        .filter { !it.hasApiFramework && it.resources.isNotEmpty() }
        .sortedWith(getCompanionComparatorBestFirst(screenData.widthPx, screenData.heightPx))
        .firstOrNull()
        ?.run {
            com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.RenderCompanion(
                // first() won't throw since "resources" value at this stage is never empty.
                resources.sortedWith(getVastResourceComparatorBestFirst()).first(),
                widthPx ?: 0,
                heightPx ?: 0,
                clicks?.clickThroughUrl,
                clicks?.clickTrackingUrls ?: emptyList(),
                creativeViewTrackingList.map { it.url }
            )
        }

    private fun tryPrepareRenderIcon(icons: List<Icon>): RenderIcon? = icons
        .filter { !it.hasApiFramework }
        .sortedWith(getIconComparatorBestFirst())
        .firstOrNull()
        ?.run {
            com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.RenderIcon(
                resource,
                widthPx ?: 0,
                heightPx ?: 0,
                clicks?.clickThroughUrl,
                clicks?.clickTrackingUrlList ?: emptyList(),
                viewTrackingUrlList,
                durationMillis,
                offset
            )
        }

    private companion object VAST {

        private const val WRAPPER_TIMEOUT_MILLIS = 5000L
        private const val WRAPPER_DEPTH_MAX = 4

        private fun isCompatibleLinearType(type: String): Boolean = type.lowercase().let {
            it == "video/mp4" || it == "video/3gpp" || it == "video/webm"
        }

        private fun createLinearTracking(
            trackingList: List<Tracking>,
            videoClicks: VideoClicks?
        ): LinearTracking {
            val tracking = trackingList.groupBy { it.event }

            return LinearTracking(
                videoClicks.toClickTrackUrls(),
                tracking.toUrls(TrackingEvent.CreativeView),
                tracking.toUrls(TrackingEvent.Start),
                tracking.toUrls(TrackingEvent.FirstQuartile),
                tracking.toUrls(TrackingEvent.Midpoint),
                tracking.toUrls(TrackingEvent.ThirdQuartile),
                tracking.toUrls(TrackingEvent.Complete),
                tracking.toUrls(TrackingEvent.Mute),
                tracking.toUrls(TrackingEvent.UnMute),
                tracking.toUrls(TrackingEvent.Pause),
                tracking.toUrls(TrackingEvent.Resume),
                tracking.toUrls(TrackingEvent.Rewind),
                tracking.toUrls(TrackingEvent.Skip),
                tracking.toUrls(TrackingEvent.CloseLinear),
                tracking.toLinearProgressTracking()
            )
        }

        private fun Map<TrackingEvent, List<Tracking>>.toUrls(
            eventType: TrackingEvent
        ): List<String> = this[eventType]?.map { it.url } ?: emptyList()

        private fun Map<TrackingEvent, List<Tracking>>.toLinearProgressTracking(): List<LinearProgressTracking> =
            this[TrackingEvent.Progress]?.mapNotNull {
                if (it.offset == null) null else LinearProgressTracking(it.url, it.offset)
            } ?: emptyList()

        private fun VideoClicks?.toClickTrackUrls(): List<String> =
            this?.clickTrackingList?.map { it.url } ?: emptyList()

        // We do not support VPAID and whatever for now.
        private val Creative.hasApiFramework
            get() = !apiFramework.isNullOrBlank()

        private val Companion.hasApiFramework
            get() = !apiFramework.isNullOrBlank()

        private val MediaFile.hasApiFramework
            get() = !apiFramework.isNullOrBlank()

        private val Icon.hasApiFramework
            get() = !apiFramework.isNullOrBlank()

        // TODO. Move to Set.kt?
        private operator fun <T> Set<T>?.plus(item: T?): Set<T> {
            val newSet = mutableSetOf<T>()

            this?.let { newSet += it }
            item?.let { newSet += it }

            return newSet
        }

        // TODO. Move to List.kt?
        private operator fun <T> List<T>?.plus(items: List<T>?): List<T> {
            val newList = mutableListOf<T>()

            this?.let { newList += it }
            items?.let { newList += it }

            return newList
        }

        // TODO. Move to List.kt?
        private operator fun <T> List<T>?.plus(item: T?): List<T> =
            item?.let { this + listOf(it) } ?: this ?: emptyList()
    }
}
