package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast

import android.util.Xml
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.Result
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.createSimpleDateFormat
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdLoadError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Ad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.AdChild
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.AdParameters
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.AdSystem
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Companion
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.CompanionClicks
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Creative
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.CreativeChild
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.HtmlResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.IFrameResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Icon
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.IconClicks
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Impression
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.InLine
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Linear
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.MediaFile
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Offset
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Pricing
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.StaticResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Tracking
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.TrackingEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Vast
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VideoClick
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VideoClicks
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Wrapper
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.ensureActive
import kotlinx.coroutines.withContext
import org.xmlpull.v1.XmlPullParser
import org.xmlpull.v1.XmlPullParserException
import java.io.StringReader
import java.text.NumberFormat

fun VastParser(): VastParser = Instance

private val Instance by lazy {
    VastParserImpl()
}

typealias VastParserResult = Result<Vast, VastAdLoadError>

interface VastParser {
    suspend operator fun invoke(vastXml: String): VastParserResult
}

// TODO. Implement so that each tag has its own handler + absolute depth support.
//  The current implementation is ugly as hell.
private class VastParserImpl : VastParser {
    override suspend fun invoke(vastXml: String): VastParserResult = try {
        StringReader(vastXml).use { vastXmlReader ->
            val parser = Xml.newPullParser().apply {
                setFeature(XmlPullParser.FEATURE_PROCESS_NAMESPACES, false)
                setInput(vastXmlReader)
            }

            val parsedVast = parser.parseVast()
            if (parsedVast != null) {
                Result.Success(parsedVast)
            } else {
                Result.Failure(VastAdLoadError.VAST_AD_LOAD_NO_XML_TAG_ERROR)
            }
        }
    } catch (e: Exception) {
        Result.Failure(VastAdLoadError.VAST_AD_LOAD_XML_PARSE_ERROR)
    }
}

// Once tag is closed and we need to go to the parent one, return from the function.
private suspend inline fun XmlPullParser.iterateTag(
    crossinline onCurrentTag: suspend XmlPullParser.() -> Unit,
    crossinline onCurrentText: suspend XmlPullParser.(trimmedText: String) -> Unit,
    crossinline onDirectChildTag: suspend XmlPullParser.() -> Unit
): Unit = coroutineScope {
    ensureActive()

    // Trying to move to the first start tag of the xml document.
    if (isStartDocument) nextTag()

    if (isEndDocument) return@coroutineScope

    if (!isStartTag) {
        throw XmlPullParserException(
            "iterateCurrentTagEvents call is allowed only for START_TAG event"
        )
    }

    val initialDepth = depth
    // Go to the inner tags as well, but not to the outside.
    while (depth >= initialDepth) {
        when (depth - initialDepth) {
            // Initial tag depth.
            0 -> when {
                isStartTag -> onCurrentTag()
                isText && !text.isNullOrBlank() -> onCurrentText(text.trim())
                isEndTag -> {
                    // Leaving the initial tag completely.
                    return@coroutineScope
                }
            }
            // Direct child tag depth.
            1 -> if (isStartTag) {
                onDirectChildTag()
            }
        }

        next()
    }
}

private fun XmlPullParser.attribute(name: String) =
    getAttributeValue(null, name)?.takeIf { it.isNotBlank() }

private val XmlPullParser.isStartDocument get() = eventType == XmlPullParser.START_DOCUMENT
private val XmlPullParser.isStartTag get() = eventType == XmlPullParser.START_TAG
private val XmlPullParser.isText get() = eventType == XmlPullParser.TEXT
private val XmlPullParser.isEndTag get() = eventType == XmlPullParser.END_TAG
private val XmlPullParser.isEndDocument get() = eventType == XmlPullParser.END_DOCUMENT

// I'm using IO since next() call blocks thread and probably it's IO operation.
private suspend fun XmlPullParser.parseVast(): Vast? = withContext(DispatcherProvider().io) {
    parseVastTag()
}

private suspend fun XmlPullParser.parseVastTag(): Vast? {
    val ads = mutableListOf<Ad>()
    var version: String? = null
    var errorUrl: String? = null

    // Assuming, the first and the only root tag is <VAST>.
    // If someone decides to put multiple root tags: I don't care since it isn't by XML rules.
    iterateTag(
        onCurrentTag = {
            version = attribute("version")
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_ERROR -> errorUrl = parseErrorTag()
                TAG_AD -> parseAdTag()?.let { ads.add(it) }
            }
        }
    )

    return if (ads.isEmpty() && errorUrl == null) null else Vast(ads, errorUrl, version)
}

private suspend fun XmlPullParser.parseAdTag(): Ad? {
    var id: String? = null
    var sequence: Int? = null

    var adChild: AdChild? = null

    iterateTag(
        onCurrentTag = {
            id = attribute("id")
            sequence = parseSequenceAttribute()
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (adChild == null) {
                adChild = when (name) {
                    TAG_INLINE -> AdChild.InLine(parseInLineTag())
                    TAG_WRAPPER -> parseWrapperTag()?.let { AdChild.Wrapper(it) }
                    else -> null
                }
            }
        }
    )

    return adChild?.let { Ad(id, sequence, it) }
}

private suspend fun XmlPullParser.parseInLineTag(): InLine {
    var adSystem: AdSystem? = null
    var adTitle: String? = null
    var description: String? = null
    var advertiser: String? = null
    var pricing: Pricing? = null
    val impressions = mutableListOf<Impression>()
    val errorUrls = mutableListOf<String>()
    val creatives = mutableListOf<Creative>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_AD_SYSTEM -> adSystem = parseAdSystemTag()
                TAG_AD_TITLE -> adTitle = parseTagsTextOnly()
                TAG_DESCRIPTION -> description = parseTagsTextOnly()
                TAG_ADVERTISER -> advertiser = parseTagsTextOnly()
                TAG_PRICING -> pricing = parsePricingTag()
                TAG_IMPRESSION -> parseImpressionTag()?.let { impressions.add(it) }
                TAG_ERROR -> parseErrorTag()?.let { errorUrls.add(it) }
                TAG_CREATIVES -> parseCreativesTag(
                    isInLineParent = true
                ).let { creatives.addAll(it) }
            }
        }
    )

    return InLine(
        adSystem,
        adTitle,
        description,
        advertiser,
        pricing,
        impressions,
        errorUrls,
        creatives
    )
}

private suspend fun XmlPullParser.parseWrapperTag(): Wrapper? {
    var vastAdTagUrl: String? = null
    var followAdditionalWrappers: Boolean? = null
    var adSystem: AdSystem? = null
    val impressions = mutableListOf<Impression>()
    val errorUrls = mutableListOf<String>()
    val creatives = mutableListOf<Creative>()

    iterateTag(
        onCurrentTag = {
            followAdditionalWrappers =
                attribute("followAdditionalWrappers")?.let { it.toBoolean() }
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_VAST_AD_TAG_URI -> vastAdTagUrl = parseTagsTextOnly()
                TAG_AD_SYSTEM -> adSystem = parseAdSystemTag()
                TAG_IMPRESSION -> parseImpressionTag()?.let { impressions.add(it) }
                TAG_ERROR -> parseErrorTag()?.let { errorUrls.add(it) }
                TAG_CREATIVES -> parseCreativesTag(
                    isInLineParent = false
                ).let { creatives.addAll(it) }
            }
        }
    )

    return vastAdTagUrl?.let {
        Wrapper(
            it,
            followAdditionalWrappers,
            adSystem,
            impressions,
            errorUrls,
            creatives
        )
    }
}

private fun XmlPullParser.parseSequenceAttribute(): Int? =
    attribute("sequence")?.let { it.toIntOrNull() ?: 999 }

private suspend fun XmlPullParser.parseAdSystemTag(): AdSystem? {
    var name: String? = null
    var version: String? = null

    iterateTag(
        onCurrentTag = {
            version = attribute("version")
        },
        onCurrentText = { trimmedText ->
            name = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return if (name == null && version == null) null else AdSystem(name, version)
}

private suspend fun XmlPullParser.parseErrorTag() = parseTagsTextOnly()

private suspend fun XmlPullParser.parseTagsTextOnly(): String? {
    var simpleText: String? = null

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = { trimmedText ->
            simpleText = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return simpleText
}

private suspend fun XmlPullParser.parsePricingTag(): Pricing? {
    var model: String? = null
    var currency: String? = null

    iterateTag(
        onCurrentTag = {
            model = attribute("model")
            currency = attribute("currency")
        },
        onCurrentText = {
        },
        onDirectChildTag = {
        }
    )

    return if (model == null && currency == null) null else Pricing(model, currency)
}

private suspend fun XmlPullParser.parseImpressionTag(): Impression? {
    var id: String? = null
    var impressionUrl: String? = null

    iterateTag(
        onCurrentTag = {
            id = attribute("id")
        },
        onCurrentText = { trimmedText ->
            impressionUrl = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return impressionUrl?.let { Impression(id, it) }
}

private suspend fun XmlPullParser.parseCreativesTag(isInLineParent: Boolean): List<Creative> {
    val creatives = mutableListOf<Creative>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (name == TAG_CREATIVE) {
                parseCreativeTag(isInLineParent)?.let { creatives.add(it) }
            }
        }
    )

    return creatives
}

private suspend fun XmlPullParser.parseCreativeTag(isInLineParent: Boolean): Creative? {
    var id: String? = null
    var sequence: Int? = null
    var adId: String? = null
    var apiFramework: String? = null
    var child: CreativeChild? = null

    iterateTag(
        onCurrentTag = {
            id = attribute("id")
            sequence = parseSequenceAttribute()
            adId = attribute("adID")
            apiFramework = attribute("apiFramework")
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (child == null) {
                child = when (name) {
                    TAG_LINEAR -> {
                        parseLinearTag(isInLineParent)
                            ?.let { CreativeChild.Linear(it) }
                    }
                    TAG_COMPANION_ADS -> {
                        parseCompanionAdsTag()
                            .takeIf { it.isNotEmpty() }
                            ?.let { CreativeChild.Companions(it) }
                    }
                    else -> null
                }
            }
        }
    )

    return child?.let { Creative(id, sequence, adId, apiFramework, it) }
}

private suspend fun XmlPullParser.parseLinearTag(isInLineParent: Boolean): Linear? {
    var skipOffset: Offset? = null
    var adParameters: AdParameters? = null
    var durationMillis: Long? = null
    val mediaFiles = mutableListOf<MediaFile>()
    val trackingList = mutableListOf<Tracking>()
    var videoClicks: VideoClicks? = null
    val icons = mutableListOf<Icon>()

    iterateTag(
        onCurrentTag = {
            skipOffset = parseSkipOffsetAttribute()
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_AD_PARAMETERS -> adParameters = parseAdParametersTag()
                TAG_DURATION -> durationMillis = parseTagsTextOnly()?.toMillis()
                TAG_MEDIA_FILES -> mediaFiles.addAll(parseMediaFilesTag())
                TAG_TRACKING_EVENTS -> trackingList.addAll(parseTrackingEventsTag())
                TAG_VIDEO_CLICKS -> videoClicks = parseVideoClicksTag(isInLineParent)
                TAG_ICONS -> icons.addAll(parseIconsTag())
            }
        }
    )

    return if (isInLineParent && mediaFiles.isEmpty()) {
        null
    } else {
        Linear(
            skipOffset,
            adParameters,
            durationMillis,
            mediaFiles,
            trackingList,
            videoClicks,
            icons
        )
    }
}

private fun XmlPullParser.parseSkipOffsetAttribute(): Offset? =
    attribute("skipoffset")?.toOffset()

private fun String.toOffset(): Offset? = toMillis()?.let { Offset.Time(it) }
    ?: toPercents()?.let { Offset.Percents(it) }

// TODO. Not sure if that's the best way.
private fun String.toMillis(): Long? = kotlin.runCatching {
    timeFormatterWithMillis().parse(this)?.time
}.getOrNull() ?: kotlin.runCatching {
    timeFormatterWithoutMillis().parse(this)?.time
}.getOrNull()

private fun timeFormatterWithoutMillis() = createSimpleDateFormat("HH:mm:ss")

private fun timeFormatterWithMillis() = createSimpleDateFormat("HH:mm:ss.SSS")

// TODO. Not sure if that's the best way.
private fun String.toPercents(): Int? = kotlin.runCatching {
    percentageFormatter.parse(this)
        ?.let { (it.toFloat() * 100).toInt() }
        ?.takeIf { it in 0..100 }
}.getOrNull()

private val percentageFormatter = NumberFormat.getPercentInstance()

private suspend fun XmlPullParser.parseAdParametersTag(): AdParameters? {
    var rawText: String? = null
    var isXmlEncoded: Boolean? = null

    iterateTag(
        onCurrentTag = {
            isXmlEncoded = attribute("xmlEncoded")?.let { it.toBoolean() }
        },
        onCurrentText = { trimmedText ->
            rawText = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return rawText?.let { AdParameters(it, isXmlEncoded) }
}

private suspend fun XmlPullParser.parseMediaFilesTag(): List<MediaFile> {
    val mediaFiles = mutableListOf<MediaFile>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (name == TAG_MEDIA_FILE) {
                parseMediaFileTag()?.let { mediaFiles.add(it) }
            }
        }
    )

    return mediaFiles
}

private suspend fun XmlPullParser.parseMediaFileTag(): MediaFile? {
    var mediaFileUrl: String? = null
    var id: String? = null
    var isProgressiveDelivery: Boolean? = null
    var type: String? = null
    var widthPx: Int? = null
    var heightPx: Int? = null
    var codec: String? = null
    var bitrate: Int? = null
    var minBitrate: Int? = null
    var maxBitrate: Int? = null
    var isScalable: Boolean? = null
    var apiFramework: String? = null

    iterateTag(
        onCurrentTag = {
            id = attribute("id")
            isProgressiveDelivery = attribute("delivery") == "progressive"
            type = attribute("type")
            widthPx = attribute("width")?.toIntOrNull()
            heightPx = attribute("height")?.toIntOrNull()
            codec = attribute("codec")
            bitrate = attribute("bitrate")?.toIntOrNull()
            minBitrate = attribute("minBitrate")?.toIntOrNull()
            maxBitrate = attribute("maxBitrate")?.toIntOrNull()
            isScalable = attribute("scalable")?.toBoolean()
            apiFramework = attribute("apiFramework")
        },
        onCurrentText = { trimmedText ->
            mediaFileUrl = trimmedText
        },
        onDirectChildTag = {
        }
    )

    // Not a fan of "!!" operator but it's better than creating multiple duplicates of the variables with weird names.
    return if (mediaFileUrl == null || type == null || isProgressiveDelivery == null) {
        null
    } else {
        MediaFile(
            mediaFileUrl!!,
            id,
            isProgressiveDelivery!!,
            type!!,
            widthPx,
            heightPx,
            codec,
            bitrate,
            minBitrate,
            maxBitrate,
            isScalable,
            apiFramework
        )
    }
}

private suspend fun XmlPullParser.parseTrackingEventsTag(): List<Tracking> {
    val trackingList = mutableListOf<Tracking>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (name == TAG_TRACKING) {
                parseTrackingTag()?.let { trackingList.add(it) }
            }
        }
    )

    return trackingList
}

private suspend fun XmlPullParser.parseTrackingTag(): Tracking? {
    var event: TrackingEvent? = null
    var url: String? = null
    var offset: Offset? = null

    iterateTag(
        onCurrentTag = {
            event = attribute("event")?.toTrackingEvent()
            offset = attribute("offset")?.toOffset()
        },
        onCurrentText = { trimmedText ->
            url = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return if (
        event == null ||
        url == null ||
        event == TrackingEvent.Progress && offset == null
    ) {
        null
    } else {
        Tracking(event!!, url!!, offset)
    }
}

// TODO. Refactor.
private fun String.toTrackingEvent(): TrackingEvent? = when (this) {
    "creativeView" -> TrackingEvent.CreativeView
    "start" -> TrackingEvent.Start
    "firstQuartile" -> TrackingEvent.FirstQuartile
    "midpoint" -> TrackingEvent.Midpoint
    "thirdQuartile" -> TrackingEvent.ThirdQuartile
    "complete" -> TrackingEvent.Complete
    "mute" -> TrackingEvent.Mute
    "unmute" -> TrackingEvent.UnMute
    "pause" -> TrackingEvent.Pause
    "rewind" -> TrackingEvent.Rewind
    "resume" -> TrackingEvent.Resume
    "closeLinear" -> TrackingEvent.CloseLinear
    "skip" -> TrackingEvent.Skip
    "progress" -> TrackingEvent.Progress
    else -> null
}

private suspend fun XmlPullParser.parseVideoClicksTag(isInLineParent: Boolean): VideoClicks? {
    var clickThrough: VideoClick? = null
    val clickTrackingList = mutableListOf<VideoClick>()
    val customClickList = mutableListOf<VideoClick>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_CLICK_THROUGH -> clickThrough = parseVideoClickTag()
                TAG_CLICK_TRACKING -> parseVideoClickTag()?.let { clickTrackingList.add(it) }
                TAG_CUSTOM_CLICK -> parseVideoClickTag()?.let { customClickList.add(it) }
            }
        }
    )

    return if (isInLineParent && clickThrough == null) {
        null
    } else {
        VideoClicks(
            clickThrough,
            clickTrackingList,
            customClickList
        )
    }
}

private suspend fun XmlPullParser.parseVideoClickTag(): VideoClick? {
    var id: String? = null
    var url: String? = null

    iterateTag(
        onCurrentTag = {
            id = attribute("id")
        },
        onCurrentText = { trimmedText ->
            url = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return url?.let { VideoClick(id, it) }
}

private suspend fun XmlPullParser.parseIconsTag(): List<Icon> {
    val icons = mutableListOf<Icon>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (name == TAG_ICON) {
                parseIconTag()?.let { icons.add(it) }
            }
        }
    )

    return icons
}

private suspend fun XmlPullParser.parseIconTag(): Icon? {
    var program: String? = null
    var widthPx: Int? = null
    var heightPx: Int? = null
    var apiFramework: String? = null
    var offset: Offset? = null
    var durationMillis: Long? = null
    var clicks: IconClicks? = null
    var resource: VastResource? = null
    val viewTrackingUrlList = mutableListOf<String>()

    iterateTag(
        onCurrentTag = {
            program = attribute("program")
            widthPx = attribute("width")?.toIntOrNull()
            heightPx = attribute("height")?.toIntOrNull()
            apiFramework = attribute("apiFramework")
            offset = attribute("offset")?.toOffset()
            durationMillis = attribute("duration")?.toMillis()
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (resource == null) {
                resource = when (name) {
                    TAG_STATIC_RESOURCE -> parseStaticResourceTag()?.let { VastResource.Static(it) }
                    TAG_HTML_RESOURCE -> parseHtmlResourceTag()?.let { VastResource.Html(it) }
                    TAG_IFRAME_RESOURCE -> parseIFrameResourceTag()?.let { VastResource.IFrame(it) }
                    else -> null
                }
            }

            when (name) {
                TAG_ICON_CLICKS -> clicks = parseIconClicksTag()
                TAG_ICON_VIEW_TRACKING -> parseTagsTextOnly()?.let { viewTrackingUrlList.add(it) }
            }
        }
    )

    return resource?.let {
        Icon(
            program,
            widthPx,
            heightPx,
            apiFramework,
            offset,
            durationMillis,
            clicks,
            viewTrackingUrlList,
            it
        )
    }
}

private suspend fun XmlPullParser.parseIconClicksTag(): IconClicks? {
    var clickThroughUrl: String? = null
    val clickTrackingUrlList = mutableListOf<String>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_ICON_CLICK_THROUGH -> clickThroughUrl = parseTagsTextOnly()
                TAG_ICON_CLICK_TRACKING -> parseTagsTextOnly()?.let { clickTrackingUrlList.add(it) }
            }
        }
    )

    return clickThroughUrl?.let {
        IconClicks(
            it,
            clickTrackingUrlList
        )
    }
}

private suspend fun XmlPullParser.parseStaticResourceTag(): StaticResource? {
    var resource: String? = null
    var creativeType: CreativeType? = null

    iterateTag(
        onCurrentTag = {
            creativeType = attribute("creativeType")?.toCreativeType()
        },
        onCurrentText = { trimmedText ->
            resource = trimmedText
        },
        onDirectChildTag = {
        }
    )

    return if (resource == null || creativeType == null) {
        null
    } else {
        StaticResource(
            resource!!,
            creativeType!!
        )
    }
}

private fun String.toCreativeType(): CreativeType? = when {
    startsWith("image/", ignoreCase = true) -> CreativeType.Image
    contains("javascript", ignoreCase = true) -> CreativeType.JS
    else -> null
}

private suspend fun XmlPullParser.parseHtmlResourceTag(): HtmlResource? =
    parseTagsTextOnly()?.let { HtmlResource(it) }

private suspend fun XmlPullParser.parseIFrameResourceTag(): IFrameResource? =
    parseTagsTextOnly()?.let { IFrameResource(it) }

private suspend fun XmlPullParser.parseCompanionAdsTag(): List<Companion> {
    val companions = mutableListOf<Companion>()

    iterateTag(
        onCurrentTag = {
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            if (name == TAG_COMPANION) {
                parseCompanionTag()?.let { companions.add(it) }
            }
        }
    )

    return companions
}

private suspend fun XmlPullParser.parseCompanionTag(): Companion? {
    var id: String? = null
    var widthPx: Int? = null
    var heightPx: Int? = null
    var altText: String? = null
    var apiFramework: String? = null
    var adParameters: AdParameters? = null
    val creativeViewTrackingList = mutableListOf<Tracking>()
    val resources = mutableListOf<VastResource>()

    val clicks: CompanionClicks?
    var clickThroughUrl: String? = null
    val clickTrackingList = mutableListOf<String>()

    // for tracking filter creativeview only.

    iterateTag(
        onCurrentTag = {
            id = attribute("id")
            widthPx = attribute("width")?.toIntOrNull()
            heightPx = attribute("height")?.toIntOrNull()
            apiFramework = attribute("apiFramework")
        },
        onCurrentText = {
        },
        onDirectChildTag = {
            when (name) {
                TAG_STATIC_RESOURCE -> {
                    parseStaticResourceTag()?.let { resources.add(VastResource.Static(it)) }
                }
                TAG_HTML_RESOURCE -> {
                    parseHtmlResourceTag()?.let { resources.add(VastResource.Html(it)) }
                }
                TAG_IFRAME_RESOURCE -> {
                    parseIFrameResourceTag()?.let { resources.add(VastResource.IFrame(it)) }
                }

                TAG_ALT_TEXT -> altText = parseTagsTextOnly()
                TAG_AD_PARAMETERS -> adParameters = parseAdParametersTag()

                TAG_TRACKING_EVENTS -> {
                    creativeViewTrackingList.addAll(
                        parseTrackingEventsTag()
                            .filter { it.event == TrackingEvent.CreativeView }
                    )
                }

                TAG_COMPANION_CLICK_THROUGH -> clickThroughUrl = parseTagsTextOnly()
                TAG_COMPANION_CLICK_TRACKING -> parseTagsTextOnly()?.let {
                    clickTrackingList.add(
                        it
                    )
                }
            }
        }
    )

    clicks = clickThroughUrl?.let {
        CompanionClicks(it, clickTrackingList)
    }

    return if (resources.isEmpty()) {
        null
    } else {
        Companion(
            id,
            widthPx,
            heightPx,
            altText,
            apiFramework,
            clicks,
            adParameters,
            creativeViewTrackingList,
            resources
        )
    }
}

private const val TAG_ERROR = "Error"
private const val TAG_AD = "Ad"
private const val TAG_INLINE = "InLine"
private const val TAG_WRAPPER = "Wrapper"
private const val TAG_VAST_AD_TAG_URI = "VASTAdTagURI"
private const val TAG_AD_SYSTEM = "AdSystem"
private const val TAG_AD_TITLE = "AdTitle"
private const val TAG_DESCRIPTION = "Description"
private const val TAG_ADVERTISER = "Advertiser"
private const val TAG_PRICING = "Pricing"
@Suppress("unused")
private const val TAG_SURVEY = "Survey"
private const val TAG_IMPRESSION = "Impression"
private const val TAG_CREATIVES = "Creatives"
private const val TAG_CREATIVE = "Creative"
private const val TAG_LINEAR = "Linear"
private const val TAG_AD_PARAMETERS = "AdParameters"
private const val TAG_DURATION = "Duration"
private const val TAG_MEDIA_FILES = "MediaFiles"
private const val TAG_MEDIA_FILE = "MediaFile"
private const val TAG_TRACKING_EVENTS = "TrackingEvents"
private const val TAG_TRACKING = "Tracking"
private const val TAG_VIDEO_CLICKS = "VideoClicks"
private const val TAG_CLICK_THROUGH = "ClickThrough"
private const val TAG_CLICK_TRACKING = "ClickTracking"
private const val TAG_CUSTOM_CLICK = "CustomClick"
private const val TAG_ICONS = "Icons"
private const val TAG_ICON = "Icon"
private const val TAG_STATIC_RESOURCE = "StaticResource"
private const val TAG_IFRAME_RESOURCE = "IFrameResource"
private const val TAG_HTML_RESOURCE = "HTMLResource"
private const val TAG_ICON_CLICKS = "IconClicks"
private const val TAG_ICON_CLICK_THROUGH = "IconClickThrough"
private const val TAG_ICON_CLICK_TRACKING = "IconClickTracking"
private const val TAG_ICON_VIEW_TRACKING = "IconViewTracking"
private const val TAG_ALT_TEXT = "AltText"
private const val TAG_COMPANION_ADS = "CompanionAds"
private const val TAG_COMPANION = "Companion"
private const val TAG_COMPANION_CLICK_THROUGH = "CompanionClickThrough"
private const val TAG_COMPANION_CLICK_TRACKING = "CompanionClickTracking"
