package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose

import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.tween
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.height
import androidx.compose.material.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.geometry.CornerRadius
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress

@Composable
fun VastProgressBar(
    isPlaying: Boolean,
    progress: PlaybackProgress,
    modifier: Modifier = Modifier,
    color: Color = MaterialTheme.colors.primary
) {
    val cornerRadius = 4.dp.value

    var maxProgressRatio by rememberSaveable {
        mutableStateOf(0f)
    }

    val playbackProgress = Math.max(progress.asRatio, maxProgressRatio)
    val progressRatio = remember { Animatable(playbackProgress) }
    maxProgressRatio = progressRatio.value

    LaunchedEffect(key1 = progress, key2 = isPlaying) {
        progressRatio.snapTo(playbackProgress)
        if (isPlaying && progress is PlaybackProgress.Position) {
            progressRatio.animateTo(
                targetValue = 1f,
                animationSpec = tween(
                    durationMillis = progress.remainingMillis,
                    easing = LinearEasing
                )
            )
        } else {
            progressRatio.stop()
        }
    }
    
    Box(
        modifier
            .height(4.dp)
            .drawBehind {
                drawRoundRect(
                    size = Size(progressRatio.value * size.width, size.height),
                    color = color,
                    cornerRadius = CornerRadius(cornerRadius, cornerRadius)
                )
            }
    )
}

private val PlaybackProgress.asRatio: Float
    get() = when (this) {
        is PlaybackProgress.Finished -> 100f
        is PlaybackProgress.Position -> {
            if (totalDurationMillis == 0L) {
                0f
            } else {
                (currentPositionMillis / totalDurationMillis.toDouble()).toFloat()
            }
        }
        PlaybackProgress.NotAvailable -> 0f
    }

private val PlaybackProgress.remainingMillis: Int
    get() = when (this) {
        is PlaybackProgress.Finished -> 0
        is PlaybackProgress.Position -> {
            if (totalDurationMillis == 0L) {
                0
            } else {
                (totalDurationMillis - currentPositionMillis).toInt().coerceAtLeast(0)
            }
        }
        PlaybackProgress.NotAvailable -> 0
    }

@Composable
@Preview
private fun Preview() {
    MaterialTheme {
        VastProgressBar(
            true,
            PlaybackProgress.Position(
                currentPositionMillis = 10,
                totalDurationMillis = 100
            )
        )
    }
}
