package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear

import android.content.Context
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdShowError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.ScreenUtils.toPosition
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Offset
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Linear
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PreparedVastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ValueWrapper
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.VastPrivacyIcon
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button.ButtonType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

internal val NO_POSITION = androidx.compose.ui.geometry.Offset.Zero.toPosition()
internal class LinearControllerImpl(
    private val linear: Linear,
    positionMillis: Int,
    mute: Boolean,
    overrideSkipEnabled: Boolean?,
    overrideSkipEnabledDelaySeconds: Int,
    // Moloco feature: if user clicks skip button, open clicktrough link without VAST click tracking.
    private val autoStoreOnSkip: Boolean,
    // Moloco feature: if video ends, open clicktrough link without VAST click tracking.
    private val autoStoreOnComplete: Boolean,
    context: Context,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val externalLinkHandler: ExternalLinkHandler
) : LinearController {

    private val scope = CoroutineScope(DispatcherProvider().main)

    private val _event = MutableSharedFlow<LinearControllerEvent>()
    override val event: Flow<LinearControllerEvent> = _event

    private fun onEvent(event: LinearControllerEvent) = scope.launch { _event.emit(event) }

    private val assetUri: String = linear.networkMediaResource

    private val _mute = MutableStateFlow(mute)
    override val mute: StateFlow<Boolean> = _mute
    override fun onMuteChange(mute: Boolean) {
        _mute.value = mute

        tracker.let {
            if (mute) {
                it.trackMute(positionMillis, assetUri)
            } else {
                it.trackUnMute(positionMillis, assetUri)
            }
        }
    }

    private val _startFromMillis = MutableStateFlow(ValueWrapper(positionMillis.toLong()))
    override val startFromMillis = _startFromMillis.asStateFlow()

    override val isStreamingEnabled = SdkObjectFactory.Media.mediaConfigSingleton.isStreamingEnabled
    override val videoUri: String = if (isStreamingEnabled) linear.networkMediaResource else linear.localMediaResource.absolutePath

    override fun onButtonRendered(button: Button) {
        tracker.trackButtonRender(button)
    }

    override fun onButtonUnRendered(buttonType: ButtonType) {
        tracker.trackButtonUnrender(buttonType)
    }

    override val hasClickThrough: Boolean = linear.clickThroughUrl != null

    override fun onClickThrough(position: CustomUserEventBuilderService.UserInteraction.Position) {
        onClickThrough(track = true, position = position)
    }

    private fun onClickThrough(
        track: Boolean,
        position: CustomUserEventBuilderService.UserInteraction.Position
    ) {
        linear.clickThroughUrl?.let {
            if (track) {
                tracker.trackClick(position, positionMillis, assetUri)
            }
            externalLinkHandler(it)
            onEvent(LinearControllerEvent.ClickThrough)
        }
    }

    private val iconTracker = with(linear) {
        IconVastTracker(
            icon?.clickTracking,
            icon?.viewTracking
        )
    }

    private val vastPrivacyIconImpl = with(linear) {
        VastPrivacyIcon(
            icon?.resource,
            icon?.widthPx,
            icon?.heightPx,
            icon?.clickThroughUrl,
            scope,
            context,
            customUserEventBuilderService,
            externalLinkHandler,
            trackClick = {
                iconTracker.trackClick(
                    this@LinearControllerImpl.positionMillis,
                    assetUri
                )
            },
            trackDisplay = {
                iconTracker.trackView(
                    this@LinearControllerImpl.positionMillis,
                    assetUri
                )
            }
        )
    }

    private val shouldShowIcon = MutableStateFlow(false)

    override val vastPrivacyIcon: StateFlow<PreparedVastResource?> = combine(
        shouldShowIcon,
        vastPrivacyIconImpl.vastPrivacyIcon
    ) { shouldShowIcon, privacyIconPreparedResource ->

        if (shouldShowIcon) {
            privacyIconPreparedResource
        } else {
            null
        }
    }.stateIn(scope, SharingStarted.WhileSubscribed(), null)

    override fun onVastPrivacyIconClick() = vastPrivacyIconImpl.onVastPrivacyIconClick()

    override fun onVastPrivacyIconDisplayed() = vastPrivacyIconImpl.onVastPrivacyIconDisplayed()

    override fun onDisplayStarted() {
        onEvent(LinearControllerEvent.DisplayStarted)
    }

    override fun onVideoPlayerVisible() {
        goNextActionHolder.resume()
    }

    override fun onVideoPlayerNotVisible() {
        goNextActionHolder.pause()
    }

    override fun onIsPlaying(isPlaying: Boolean) {
        _isPlaying.value = isPlaying
    }

    private val _isPlaying = MutableStateFlow(false)
    override val isPlaying: StateFlow<Boolean> = _isPlaying

    init {
        isPlaying.onEach {
            if (it) {
                tracker.trackResume(this.positionMillis, assetUri)
            } else {
                tracker.trackPause(this.positionMillis, assetUri)
            }
        }.launchIn(scope)
    }

    private val goNextActionHolder = LinearGoNextAction(
        overrideSkipEnabled,
        overrideSkipEnabledDelaySeconds,
        linear.skipOffset
    )

    override val goNextAction by goNextActionHolder::goNextAction

    private var isSkipped = false

    override fun onSkip() {
        isSkipped = true
        tracker.trackSkip(positionMillis, assetUri)
        // TODO. Ideally, we need to update startFromMillis. Not a problem though.
        onEvent(LinearControllerEvent.Skip)

        tryFireAutoStoreOnSkip()
    }

    // Moloco feature: if user clicks skip button, open clicktrough link without VAST click tracking.
    private fun tryFireAutoStoreOnSkip() {
        if (autoStoreOnSkip) onClickThrough(track = false, NO_POSITION)
    }

    // Moloco feature: if video ends, open clicktrough link without VAST click tracking.
    private fun tryFireAutoStoreOnComplete() {
        if (autoStoreOnComplete) onClickThrough(track = false, NO_POSITION)
    }

    override fun onError(error: VastAdShowError) {
        onEvent(LinearControllerEvent.Error(error))
    }

    override fun replay() {
        // If not playing and already at the beginning - no need to trigger all the logic below.
        if (!isPlaying.value && _startFromMillis.value.value == 0L && positionMillis == 0) {
            return
        }

        _startFromMillis.value = ValueWrapper(0)
        tracker.trackRewind(positionMillis, assetUri)

        // TODO. Refactor.
        // Kind of fast state update.
        isSkipped = false
        positionMillis = 0
        goNextActionHolder.setUnavailable()
        resetIcon()
    }

    private val tracker: LinearVastTracker = LinearVastTracker.from(
        linear.tracking,
        customUserEventBuilderService
    )

    override fun destroy() {
        scope.cancel()
        vastPrivacyIconImpl.destroy()
    }

    override fun onProgress(progress: PlaybackProgress) {
        val (positionMillis, durationMillis) = when (progress) {
            is PlaybackProgress.Finished -> {
                val millis = progress.totalDurationMillis.toInt()
                millis to millis
            }
            is PlaybackProgress.Position -> {
                progress.currentPositionMillis.toInt() to progress.totalDurationMillis.toInt()
            }
            // Do nothing.
            PlaybackProgress.NotAvailable -> return
        }

        this.positionMillis = positionMillis

        if (!isSkipped) {
            tracker.trackProgress(assetUri, positionMillis, durationMillis)
        }

        if (progress is PlaybackProgress.Finished) {
            if (!isSkipped) {
                onEvent(LinearControllerEvent.Complete)

                tryFireAutoStoreOnComplete()
            }
            isSkipped = false
        }

        goNextActionHolder.updateBasedOn(positionMillis, durationMillis)
        updateIcon(positionMillis, durationMillis)
    }

    override var positionMillis: Int = 0
        private set

    private fun resetIcon() {
        shouldShowIcon.value = false
    }

    private fun updateIcon(positionMillis: Int, durationMillis: Int) {
        val icon = linear.icon ?: return

        val iconOffset = icon.offset
        val iconDurationMillis = icon.durationMillis

        val iconOffsetMillis: Int = when (iconOffset) {
            // Precision isn't really important here.
            is Offset.Percents -> durationMillis / 100 * iconOffset.percents
            is Offset.Time -> iconOffset.millis.toInt()
            else -> 0
        }.coerceIn(0..durationMillis)

        this.shouldShowIcon.value = if (iconDurationMillis == null) {
            positionMillis >= iconOffsetMillis
        } else {
            positionMillis in iconOffsetMillis..iconOffsetMillis + iconDurationMillis
        }
    }
}
