package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ResettableCountdownSecondsStateFlow
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Offset
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.GoNextAction
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch

private const val ONE_SEC = 1000L
interface LinearGoNextAction : GoNextAction {
    /**
     * Set the Linear go next action to unavailable state
     */
    fun setUnavailable()

    /**
     * Pause the Linear go next action
     */
    fun pause()

    /**
     * Resume the Linear go next action
     */
    fun resume()

    /**
     * Update the Linear go next action based on the current position and duration. (Starts the timer)
     */
    fun updateBasedOn(positionMillis: Int, durationMillis: Int)
}

internal fun LinearGoNextAction(
    overrideSkipEnabled: Boolean?,
    overrideSkipEnabledDelaySeconds: Int,
    linearSkipOffset: Offset?
): LinearGoNextAction = LinearGoNextActionImpl(
    goNextOffset = when (overrideSkipEnabled) {
        false -> null
        true -> Offset.Time(overrideSkipEnabledDelaySeconds * 1000L)
        null -> linearSkipOffset
    }
)


@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class LinearGoNextActionImpl(
    private val goNextOffset: Offset?
) : LinearGoNextAction {
    private val TAG = "LinearGoNextActionImpl"
    private val scope = CoroutineScope(DispatcherProvider().main)
    private var resettableCountdownSecondsStateFlow: ResettableCountdownSecondsStateFlow? = null

    private var timerJob: Job? = null
    private var remainingDurationSecs: UInt = 0u

    override val goNextAction = MutableStateFlow<GoNextAction.State>(
        GoNextAction.State.Unavailable
    )

    override fun pause() {
        MolocoLogger.info(TAG, "Canceling timer")
        timerJob?.cancel()
    }

    override fun resume() {
        if (remainingDurationSecs > 0u) {
            startTimer(remainingDurationSecs.toLong())
        }
    }

    override fun setUnavailable() {
        goNextAction.value = GoNextAction.State.Unavailable
    }

    override fun updateBasedOn(positionMillis: Int, durationMillis: Int) {
        val progressPercent = positionMillis / durationMillis.toDouble() * 100
        val isCreativeEnd = positionMillis >= durationMillis

        if (isCreativeEnd) {
            // Always show go next action when creative is finished.
            goNextAction.value = GoNextAction.State.Available
        } else if (goNextOffset == null) {
            MolocoLogger.info(TAG, "goNextOffset is null")
            goNextAction.value = GoNextAction.State.Unavailable
        } else {
            if (timerJob == null) {
                MolocoLogger.info(TAG, "Starting timer")
                when (goNextOffset) {
                    is Offset.Percents -> {
                        MolocoLogger.info(TAG, "Offset Percents detected")
                        startTimer(
                            remainingSeconds(
                                progressPercent.toInt()..goNextOffset.percents, durationMillis
                            ).toLong()
                        )
                    }

                    is Offset.Time -> {
                        MolocoLogger.info(TAG, "Offset Millis detected")
                        startTimer(goNextOffset.millis.millisToSeconds())
                    }
                }
            }
        }
    }

    private fun startTimer(durationSecs: Long) {
        if (timerJob.hasNotStartedOrCompleted()) {
            remainingDurationSecs = durationSecs.toUInt()
            MolocoLogger.info(TAG, "Start timer for duration: $durationSecs seconds")
            timerJob = scope.launch {
                if (resettableCountdownSecondsStateFlow == null) {
                    resettableCountdownSecondsStateFlow = ResettableCountdownSecondsStateFlow(
                        remainingDurationSecs,
                        scope
                    )
                } else {
                    resettableCountdownSecondsStateFlow?.resetTo(remainingDurationSecs)
                }

                resettableCountdownSecondsStateFlow?.countdownSecondsStateFlow?.collectLatest {
                    MolocoLogger.info(TAG, "Updating countdown to $it")
                    remainingDurationSecs = it
                    MolocoLogger.info(TAG, "Propagating state: ${it.toGoNextActionState()}")
                    goNextAction.value = it.toGoNextActionState()
                }
            }
        }
    }
}

private fun Job?.hasNotStartedOrCompleted() = this == null || this.isCancelled || this.isCompleted

private fun UInt.toGoNextActionState() = if (this == 0u) {
    GoNextAction.State.Available
} else {
    GoNextAction.State.Countdown(this)
}

private fun remainingSeconds(rangeMillis: LongRange): UInt =
    ((rangeMillis.last - rangeMillis.first) / 1000)
        .coerceAtLeast(0)
        .toUInt()

private fun remainingSeconds(rangePercents: IntRange, durationMillis: Int): UInt {
    val remainingPercents =
        (rangePercents.last - rangePercents.first).coerceAtLeast(0)

    return ((remainingPercents * durationMillis / 100.0) / 1000)
        .coerceAtLeast(0.0)
        .toUInt()
}

private fun Long.millisToSeconds() = this / ONE_SEC