package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer

import android.net.Uri
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.upstream.DataSource
import com.google.android.exoplayer2.upstream.DataSpec
import com.google.android.exoplayer2.upstream.TransferListener
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.MediaStreamStatus
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream.NotStarted
import kotlinx.coroutines.runBlocking
import java.io.RandomAccessFile
import java.io.IOException

/**
 * ProgressiveMediaFileDataSource that progressively reads from the file as it is being written.
 * This is useful for reading from a file that is being downloaded.
 */
internal class ProgressiveMediaFileDataSource(
    private val url: String,
    private val mediaCacheRepository: MediaCacheRepository
) : DataSource {
    private val TAG = "ProgressiveMediaFileDataSource"
    private var randomAccessFile: RandomAccessFile? = null
    private var bytesRemaining: Long = 0
    private var hasReadSomeBytes: Boolean = false
    var hasMediaStreamingError: Boolean = false
        private set

    override fun open(dataSpec: DataSpec): Long = runBlocking {
        MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread()}], dataSpec.length: ${dataSpec.length}, dataSpec.position: ${dataSpec.position} open: $url")
        try {
            val streamingStatus = streamingStatus(url)
            val file = when(streamingStatus) {
                is MediaStreamStatus.Complete -> {
                    MolocoLogger.info(TAG, "Complete file available for read: ${streamingStatus.file.absolutePath}")
                    streamingStatus.file
                }
                is MediaStreamStatus.InProgress -> {
                    MolocoLogger.info(TAG, "Partial file available for read: ${streamingStatus.file.absolutePath}")
                    streamingStatus.file
                }
                else -> {
                    hasMediaStreamingError = true
                    MolocoLogger.error(TAG, "Failed to download file: $url")
                    throw IOException("Cannot read file: $url")
                }
            }

            // When exoplayer tries to open the file, it should always be exist as the Vast loader
            // flow should ensure that something is downloaded.

            if (!file.exists()) {
                throw IOException("Cannot read file, does not exist yet: $url")
            }


            randomAccessFile = RandomAccessFile(file, "r").apply {
                MolocoLogger.info(TAG, "Seeked to position: ${dataSpec.position} for Opened file: ${file.absolutePath}")
                seek(dataSpec.position)
            }

            bytesRemaining = if (dataSpec.length == C.LENGTH_UNSET.toLong()) {
                MolocoLogger.info(TAG, "dataSpec length == C.LENGTH_UNSET, file.length: ${file.length()}, dataSpec.position: ${dataSpec.position}")
                file.length() - dataSpec.position
            } else {
                MolocoLogger.info(TAG, "dataSpec length != C.LENGTH_UNSET")
                dataSpec.length
            }

            if (bytesRemaining == 0L && hasStreamingLikelyFailed(streamingStatus)) {
                // This can happen if MediaCacheRepository fails to download a content
                // with some error while exoplayer has already progressivelys started playing the content.
                // In that case, we will track that it was a streaming error, so that exoplayer layer can query for it
                MolocoLogger.info(TAG, "Streaming error likely detected")
                hasMediaStreamingError = true
            }

            MolocoLogger.info(TAG, "[open] bytesRemaining: $bytesRemaining")
            return@runBlocking bytesRemaining
        } catch (e: IOException) {
            MolocoLogger.error(TAG, "Failed to open file: $url", e)
            throw e
        }
    }

    override fun read(buffer: ByteArray, offset: Int, readLength: Int): Int {
        MolocoLogger.info(TAG, "read: ${readLength}, offset: $offset")
        var bytesRead = 0
        try {
            if (readLength == 0) {
                MolocoLogger.info(TAG, "Read length is 0")
                return 0
            } else if (bytesRemaining == 0L) {
                MolocoLogger.info(TAG, "0 bytes remaining")
                return C.RESULT_END_OF_INPUT
            } else {
                val streamingStatus = streamingStatus(url)

                if (streamingStatus is MediaStreamStatus.Failure) {
                    MolocoLogger.error(TAG, "Streaming failed: $url")
                    hasMediaStreamingError = true
                    return 0
                }

                if (streamingStatus is MediaStreamStatus.Complete || streamingStatus is MediaStreamStatus.InProgress) {
                    bytesRead = randomAccessFile?.read(buffer, offset, readLength) ?: 0
                    if (streamingStatus is MediaStreamStatus.Complete) {
                        MolocoLogger.info(TAG, "streaming status: Complete, Bytes read: $bytesRead")
                    } else {
                        MolocoLogger.info(TAG, "streaming status: InProgress, Bytes read: $bytesRead")
                    }

                    if (bytesRead > 0) {
                        hasReadSomeBytes = true
                        bytesRemaining -= bytesRead
                        return bytesRead
                    }
                }
            }
        } catch (e: IOException) {
            MolocoLogger.error(TAG, "Waiting for more data", e)
        }
        return bytesRead
    }

    private fun streamingStatus(url: String) = runBlocking {
        // Having a hard time figuring out why streamMediaFaileStatus flow.first() is blocking indefinitely
        // Have to use runBlocking to work with the non-coroutine exoplayer, but then unable to retrieve
        // the value from the flow. Thus we use the snapshot API to get the value
        val status = mediaCacheRepository.streamMediaFileStatusSnapshot(url)
        MolocoLogger.info(TAG, "Collecting latest status:$status for url: $url")
        return@runBlocking status
    }

    override fun addTransferListener(transferListener: TransferListener) {
        // NO OP
        MolocoLogger.info(TAG, "addTransferListener")
    }

    override fun close() {
        MolocoLogger.info(TAG, "close")
        try {
            randomAccessFile?.close()
        } finally {
            randomAccessFile = null
        }
    }

    override fun getUri(): Uri? {
        return Uri.parse(url)
    }

    private fun hasStreamingLikelyFailed(streamingStatus: MediaStreamStatus): Boolean {
        // It is possible that the media fails to download while the streaming is in progress. This will
        // mean that MediaCacheRepository will clean the listener states as the download is no longer in progress
        // That will mean that the streaming status will go back to NotStarted. In that case, we will want to wait
        return hasReadSomeBytes && streamingStatus is MediaStreamStatus.InProgress && streamingStatus.progress == NotStarted
    }
}