package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer

import android.content.Context
import android.os.Looper
import android.view.InflateException
import androidx.annotation.MainThread
import androidx.annotation.VisibleForTesting
import androidx.lifecycle.Lifecycle
import com.google.android.exoplayer2.ExoPlayer
import com.google.android.exoplayer2.MediaItem
import com.google.android.exoplayer2.PlaybackException
import com.google.android.exoplayer2.Player
import com.google.android.exoplayer2.source.DefaultMediaSourceFactory
import com.google.android.exoplayer2.ui.StyledPlayerView
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdShowError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch


// TODO. Also make sure, all calls are on the UI thread. LaunchedEffect is UI thread by default (might change in future versions?).
//  https://developer.android.com/jetpack/compose/mental-model#parallel

/**
 * Exo player with with lifecycle awareness implementation of [VideoPlayer]
 *
 *  @property context The context of the application or activity.
 *  @property isProgressivePlayingEnabled The flag to enable progressive playing. If the media is being streamed, it will be played progressively.
 *  @property lifecycle The lifecycle associated with the player, used for managing the player's lifecycle.
 */
@MainThread
internal class SimplifiedExoPlayer(
    private val context: Context,
    private val isProgressivePlayingEnabled: Boolean,
    private val mediaCacheRepository: MediaCacheRepository,
    lifecycle: Lifecycle,
) : VideoPlayer {
    @Suppress("PrivatePropertyName")
    private val TAG = "SimplifiedExoPlayer"
    private val scope = CoroutineScope(DispatcherProvider().main)

    private val _playbackProgress =
        MutableStateFlow<PlaybackProgress>(PlaybackProgress.NotAvailable)
    override val playbackProgress: StateFlow<PlaybackProgress> = _playbackProgress

    private val _isPlaying = MutableStateFlow(PlayingState(false))
    override val isPlaying: StateFlow<PlayingState> = _isPlaying

    private val _lastError = MutableStateFlow<VastAdShowError?>(null)
    override val lastError: StateFlow<VastAdShowError?> = _lastError

    @get:MainThread
    override val playerView: StyledPlayerView? = try {
        // TryCatch block because some Chinese devices will crash when StyledPlayerView is instantiated.
        // https://mlc.atlassian.net/browse/SDK-847
        StyledPlayerView(context).apply {
            // Disables default ExoPlayer UI.
            useController = false
        }
    } catch (e: InflateException) {
        MolocoLogger.error(TAG, "ExoPlayerView could not be instantiated.", e)
        _lastError.value = VastAdShowError.VAST_AD_EXOPLAYER_STYLED_PLAYER_VIEW_INFLATE_EXCEPTION_ERROR
        null
    }

    override var uriSource: String? = null
        set(value) {
            field = value
            exoPlayer?.setUriSource(value)

            resetPlaybackState()
        }

    override var isMute: Boolean = false
        set(value) {
            field = value
            exoPlayer?.isMute = value
        }

    // ExoPlayer can be called only from one thread.
    // Compose might run @Composable function in different threads and/or in parallel,
    // that's why I'm scoping ExoPlayer to only to work with calls on the main thread.
    // https://exoplayer.dev/hello-world.html#a-note-on-threading
    // https://developer.android.com/jetpack/compose/mental-model#parallel
    private val mainThreadLooper = Looper.getMainLooper()

    @VisibleForTesting(VisibleForTesting.PRIVATE)
    @get:MainThread
    var exoPlayer: ExoPlayer? = null

    private var progressiveDataSource: ProgressiveMediaFileDataSource? = null

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    var shouldPlayVideo: Boolean = false

    init {
        isPlaying.onEach {
            if (it.isPlaying) startPlaybackProgressJob() else playbackProgressJob?.cancel()
        }.launchIn(scope)
    }

    override fun play() {
        shouldPlayVideo = true
        exoPlayer?.play()
    }

    override fun seekTo(positionMillis: Long) {
        savedSeekToPositionMillis = positionMillis
        exoPlayer?.seekTo(positionMillis)
    }

    override fun pause() {
        shouldPlayVideo = false
        exoPlayer?.pause()
    }

    override fun destroy() {
        scope.cancel()
        lifecycleHandler.destroy()
        disposeExoPlayer()
    }

    private val exoPlayerListener = object : Player.Listener {
        override fun onPlaybackStateChanged(playbackState: Int) {
            super.onPlaybackStateChanged(playbackState)
            if (playbackState == Player.STATE_ENDED) {
                onPlaybackProgress(PlaybackProgress.Finished(exoPlayer?.duration ?: 1))
                resetPlaybackState()
            }
        }

        override fun onIsPlayingChanged(isPlaying: Boolean) {
            super.onIsPlayingChanged(isPlaying)
            val hasMore = (exoPlayer?.duration ?: 0) - (exoPlayer?.currentPosition ?: 0) > 0
            _isPlaying.value = PlayingState(isPlaying = isPlaying, isVisible = true, hasMore = hasMore)
        }

        override fun onPlayerError(error: PlaybackException) {
            super.onPlayerError(error)
            MolocoLogger.error(TAG, "Exoplayer error (streaming enabled = $isProgressivePlayingEnabled)", error)
            // If the exoplayer error is due to the stream failing, check if we started any playback
            // In that case the user has viewed some of the ad and we should not show the error
            if (isProgressivePlayingEnabled && progressiveDataSource?.hasMediaStreamingError == true) {
                when(_playbackProgress.value) {
                    is PlaybackProgress.Finished,
                    is PlaybackProgress.Position -> {
                        // Ignore the error,
                        // TODO: log it in our non-fatal internal error reporting API in future
                        MolocoLogger.info(TAG, "Ignoring exoplayer streaming error as the user has viewed some of the ad already")
                        return
                    }
                    PlaybackProgress.NotAvailable -> {
                        MolocoLogger.info(TAG, "Exoplayer streaming failed before any playback started, so report that as error")
                    }
                }
            }
            _lastError.value = VastAdShowError.VAST_AD_EXOPLAYER_VIDEO_LAYER_ERROR
        }
    }

    private fun initOrResumeExoPlayer() {
        MolocoLogger.info(TAG, "Init exo player")
        // There was an error instantiating the object. So we just return
        val view: StyledPlayerView = playerView ?: return

        if (exoPlayer == null) {
            val player = ExoPlayer.Builder(context)
                .setLooper(mainThreadLooper)
                .setPauseAtEndOfMediaItems(true)
                .build()

            with(player) {
                view.player = this
                exoPlayer = this
                playWhenReady = false
                addListener(exoPlayerListener)
                restoreState()
            }
        }

        // Important for it to be at the end of exoPlayer creation.
        view.onResume()
    }

    private fun disposeExoPlayer() {
        MolocoLogger.info(TAG, "Disposing exo player")
        playerView?.run {
            // Important for it to be at the start of exoPlayer disposal.
            onPause()
            player = null
        }

        val hasMore = (exoPlayer?.duration ?: 0) - (exoPlayer?.currentPosition ?: 0) > 0

        exoPlayer?.run {
            saveState()
            removeListener(exoPlayerListener)
            release()
        }

        exoPlayer = null

        // exoPlayerListener is removed by now, therefore write conflicts should not occur.
        _isPlaying.value = PlayingState(isPlaying = false, isVisible = false, hasMore = hasMore)
    }

    private val lifecycleHandler = SimplifiedExoPlayerLifecycleHandler(
        lifecycle,
        onExoResume = this::initOrResumeExoPlayer,
        onExoPause = this::disposeExoPlayer
    )

    private fun ExoPlayer.saveState() {
        savedSeekToPositionMillis = currentPosition
    }

    private fun ExoPlayer.restoreState() {
        isMute = this@SimplifiedExoPlayer.isMute

        setUriSource(this@SimplifiedExoPlayer.uriSource)

        seekTo(savedSeekToPositionMillis)

        if (shouldPlayVideo) play() else pause()
    }

    private var ExoPlayer.isMute: Boolean
        get() = volume == 0f
        set(value) {
            volume = if (value) 0f else 1f
        }

    private fun ExoPlayer.setUriSource(uriSource: String?) {
        if (uriSource == null) {
            // It causes NPE for some reason.
            // MediaItem.EMPTY
            MolocoLogger.info(TAG, "URI Source is empty")
        } else {
            try {
                if (isProgressivePlayingEnabled) {
                    MolocoLogger.info(TAG, "Streaming is enabled")
                    val mediaSourceFactory = DefaultMediaSourceFactory {
                        ProgressiveMediaFileDataSource(
                            uriSource,
                            mediaCacheRepository,
                        ).apply {
                            progressiveDataSource = this
                        }
                    }
                    val mediaItem = MediaItem.fromUri(uriSource)
                    setMediaSource(mediaSourceFactory.createMediaSource(mediaItem))

                } else {
                    MolocoLogger.info(TAG, "Streaming is disabled")
                    setMediaItem(MediaItem.fromUri(uriSource))
                }

                prepare()
            } catch (e: Exception) {
                // setMediaItem throws exception when MediaItem can't be rendered.
                MolocoLogger.error(TAG, "ExoPlayer setMediaItem exception", e)
                _lastError.value = VastAdShowError.VAST_AD_EXOPLAYER_SET_MEDIA_ITEM_EXCEPTION_ERROR
            }
        }
    }

    // TODO. Rename. Refactor.
    private fun resetPlaybackState() {
        // Reset playback state when media item is changed.
        // player stops automatically after calling setSource.
        shouldPlayVideo = false
        savedSeekToPositionMillis = 0
        // TODO. Progress should be reset too.
    }

    // It's used:
    // 1. seeking to the specific video frame seekTo() public API method,
    // 2. exoplayer restoration stage, video player position before disposing old exoplayer instance.
    // 3. When the source is changed, we reset it's value to 0.
    // Writes to this variable by #1 or #2 use-cases in any order are allowed, since seekTo() can be called after current exoPlayer instance disposal,
    // meaning we need to seek to that new position afterwards; there's no point to keep that restoration value from #2.
    private var savedSeekToPositionMillis: Long = 0

    private fun onPlaybackProgress(value: PlaybackProgress) {
        _playbackProgress.value = value
    }

    private var playbackProgressJob: Job? = null

    private fun startPlaybackProgressJob() {
        playbackProgressJob?.cancel()
        playbackProgressJob = scope.launch {
            // ExoPlayer does not provide any progress API event AFAIK, that's why polling.
            while (true) {
                exoPlayer?.let { exo ->
                    onPlaybackProgress(
                        PlaybackProgress.Position(exo.currentPosition, exo.duration)
                    )
                }
                delay(500)
            }
        }
    }
}