package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview

import android.app.Activity
import android.content.Context
import android.view.View
import android.webkit.WebView
import androidx.activity.compose.BackHandler
import androidx.annotation.VisibleForTesting
import androidx.compose.animation.Crossfade
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.snapshotFlow
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.ComposeView
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import com.moloco.sdk.R
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.HideSystemUI
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.LifecycleAwareAdCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultAdCloseCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultWebViewAdBadge
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.theme.Theme
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ValueWrapper
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.DEC
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdGoNextButton
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button
import kotlinx.coroutines.flow.MutableStateFlow

/**
 * Typealias for a function that creates a WebView renderer.
 *
 * @param context The context in which the WebView renderer is created.
 * @param webView The WebView to display the ad content.
 * @param closeDelaySeconds The delay in seconds before the ad can be closed.
 * @param canClose Mutable state flow indicating whether the ad can be closed.
 * @param onButtonRendered Callback function when a button is rendered.
 * @param onClose Callback function when the ad is closed.
 * @param watermark Optional watermark to be displayed on the ad.
 * @return A View containing the rendered WebView.
 */
internal typealias AdWebViewRenderer = (
    context: Context,
    webView: WebView,
    closeDelaySeconds: Int,
    canClose: MutableStateFlow<Boolean>,
    onButtonRendered: (Button) -> Unit,
    onClose: () -> Unit,
    watermark: Watermark?
) -> View

/**
 * Default implementation of the WebViewRendererFactory.
 *
 * @param backgroundColor Background color of the Box containing the WebView.
 * @param adCloseCountdownButton Factory function to create an optional countdown button for the ad.
 * @return A function that creates a ComposeView containing the WebView renderer.
 */
internal fun defaultAdWebViewRenderer(
    backgroundColor: Color = Color.Black,
    adCloseCountdownButton: @Composable () -> AdCountdownButton? = { defaultAdCloseCountdownButton() },
): AdWebViewRenderer = { context, webView, closeDelaySeconds, canClose, onButtonRendered, onClose, watermark ->
    ComposeView(context).apply {
        @VisibleForTesting(otherwise = VisibleForTesting.NONE)
        id = R.id.moloco_fullscreen_ad_view_id

        setContent {
            Theme {
                val canCloseMutableState = remember {
                    mutableStateOf(canClose.value)
                }

                LaunchedEffect(Unit) {
                    snapshotFlow { canCloseMutableState.value }
                        .collect { canClose.value = it }
                }

                AdWebViewRenderDisplay(
                    webView = webView,
                    closeDelaySeconds = closeDelaySeconds,
                    canClose = canCloseMutableState,
                    onButtonRendered = onButtonRendered,
                    onClose = onClose,
                    backgroundColor = backgroundColor,
                    adCloseCountdownButton = adCloseCountdownButton(),
                    watermark = watermark
                )
            }
        }
    }
}

/**
 * Composable object to display an Ad WebView screen within an Activity.
 *
 * @param webView The WebView to display the ad content.
 * @param closeDelaySeconds The delay in seconds before the ad can be closed.
 * @param onButtonRendered Callback function when a button is rendered.
 * @param onClose Callback function when the ad is closed.
 * @param webViewRendererFactory Factory function to create the WebView renderer.
 * @param watermark Optional watermark to be displayed on the ad.
 */
@Composable
internal fun Activity.MraidAdWebViewScreen(
    adViewModel: AdViewModel,
    webView: WebView,
    closeDelaySeconds: Int,
    onButtonRendered: (Button) -> Unit,
    onClose: () -> Unit,
    adWebViewRenderer: AdWebViewRenderer,
    watermark: Watermark?,
    adCloseCountdownButton: AdCountdownButton?,
    adSkipCountdownButton: AdCountdownButton?,
) {
    Box(
        modifier = Modifier
            .fillMaxSize()
            .background(Color.Black)
    ) {
        val canCloseStateFlow = remember(closeDelaySeconds) {
            MutableStateFlow(closeDelaySeconds == 0)
        }

        fun fireOnClose() {
            if (canCloseStateFlow.value) onClose()
        }

        val currentAdPart by adViewModel.currentAdPart.collectAsState()

        Crossfade(targetState = currentAdPart) { adPart ->
            when (adPart) {
                is AdViewModel.AdPart.Companion -> {/* NO OP */ }
                is AdViewModel.AdPart.Linear -> {/* NO OP */ }
                is AdViewModel.AdPart.DEC ->
                    DEC(
                        viewModel = adPart.viewModel,
                        overrideDECOnClick = null,
                        null,
                        Modifier.fillMaxSize()
                    )
                is AdViewModel.AdPart.Mraid ->
                    AndroidView(
                        factory = {
                            adWebViewRenderer(
                                it,
                                webView,
                                closeDelaySeconds,
                                canCloseStateFlow,
                                onButtonRendered,
                                { fireOnClose() },
                                watermark
                            )
                        }
                    )
                null -> {}
            }
        }

        // responsible for transitioning the entire ad to the next ad part
        // for example: mraid ad -> DEC -> close
        AdGoNextButton(
            adViewModel,
            currentAdPart,
            adSkipCountdownButton,
            adCloseCountdownButton
        )

        BackHandler { fireOnClose() }
        HideSystemUI() // need to extend Activity for this function
    }
}

@Composable
internal fun Activity.StaticAdWebViewScreen(
    webView: WebView,
    closeDelaySeconds: Int,
    onButtonRendered: (Button) -> Unit,
    onClose: () -> Unit,
    adWebViewRenderer: AdWebViewRenderer,
    watermark: Watermark?,
) {
    Box(
        modifier = Modifier
            .fillMaxSize()
            .background(Color.Black)
    ) {

        val canCloseStateFlow = remember(closeDelaySeconds) {
            MutableStateFlow(closeDelaySeconds == 0)
        }

        fun fireOnClose() {
            if (canCloseStateFlow.value) onClose()
        }

        AndroidView(
            factory = {
                adWebViewRenderer(
                    it,
                    webView,
                    closeDelaySeconds,
                    canCloseStateFlow,
                    onButtonRendered,
                    { fireOnClose() },
                    watermark
                )
            }
        )

        BackHandler { fireOnClose() }
        HideSystemUI() // need to extend Activity for this function
    }
}

/**
 * Composable object to render a WebView with ad content.
 *
 * @param webView The WebView to display the ad content.
 * @param closeDelaySeconds The delay in seconds before the ad can be closed.
 * @param canClose Mutable state indicating whether the ad can be closed.
 * @param onButtonRendered Callback function when a button is rendered.
 * @param onClose Callback function when the ad is closed.
 * @param modifier Modifier to be applied to the Box containing the WebView.
 * @param backgroundColor Background color of the Box containing the WebView.
 * @param adCloseCountdownButton Optional countdown button to be displayed on the ad.
 * @param watermark Optional watermark to be displayed on the ad.
 */
@Composable
private fun AdWebViewRenderDisplay(
    webView: WebView,
    closeDelaySeconds: Int,
    canClose: MutableState<Boolean>,
    onButtonRendered: (button: Button) -> Unit,
    onClose: () -> Unit,
    modifier: Modifier = Modifier,
    backgroundColor: Color = Color.Black,
    adCloseCountdownButton: AdCountdownButton? = defaultAdCloseCountdownButton(),
    watermark: Watermark?
) {
    Box(
        modifier = modifier
            .fillMaxSize()
            .background(backgroundColor)
    ) {
        AdWebView(webView, Modifier.fillMaxSize(), watermark)

        adCloseCountdownButton?.let {
            val initialSecondsLeft = remember(closeDelaySeconds) {
                ValueWrapper(closeDelaySeconds.coerceAtLeast(0).toUInt())
            }

            LifecycleAwareAdCountdownButton(
                visible = true,
                onButtonRendered = onButtonRendered,
                initialSecondsLeft = initialSecondsLeft,
                canClickAfterCountdown = canClose.value,
                onCountdownFinished = { canClose.value = true },
                onClick = onClose,
                basedOnAdCountdownButton = it
            )
        }

        defaultWebViewAdBadge(
            modifier = Modifier
                .align(Alignment.BottomStart)
                .padding(12.dp))(onButtonRendered)
    }
}
