package com.moloco.sdk.xenoss.sdkdevkit.android.core.services

typealias ButtonLayoutSnapshot = List<CustomUserEventBuilderService.UserInteraction.Button>

/**
 * Custom User Event builder/provider that can build a metric event given input parameters
 */
interface CustomUserEventBuilderService {

    /**
     * Represents the different forms of user interaction that are recorded by Moloco SDK
     */
    sealed class UserInteraction {
        /**
         * Represents the position in Dp units with density/scale factor
         */
        data class Position(val topLeftXDp: Float, val topLeftYDp: Float)

        /**
         * Represents the size in Dp units with density/scale factor
         */
        data class Size(val widthDp: Float, val heightDp: Float)

        /**
         * Represents the different buttons shown in ads
         */
        data class Button(val buttonType: ButtonType, val position: Position, val size: Size) {
            // Order should match -
            // https://github.com/moloco/moloco-sdk-client-backend-shared-proto/blob/main/shared_proto/user_intent.proto#L102
            enum class ButtonType {
                NONE, // Should not be used, only to align with proto ordinals
                CLOSE,
                SKIP,
                SKIP_DEC,
                MUTE,
                UNMUTE,
                CTA,
                REPLAY,
                AD_BADGE
            }
        }

        /**
         * Impression -> user saw an ad
         */
        object Impression : UserInteraction()

        /**
         * Click -> User clicked on the ad
         * @param viewPosition is null for fullscreen ads
         * @param viewSize is null for fullscreen ads
         */
        class Click(
            val clickPosition: Position,
            val viewPosition: Position? = null,
            val viewSize: Size? = null,
            val buttonLayout: ButtonLayoutSnapshot = listOf(),
        ) : UserInteraction()

        /**
         *  AppForeground -> User navigated back to the publisher app
         */
        data class AppForeground(val lastBgTimestamp: Long) : UserInteraction()

        /**
         * AppBackground -> User navigated away from the publisher app
         */
        object AppBackground : UserInteraction()
    }

    /**
     * @return A base64 encoded protobuf string event as per definition here -
     * https://docs.google.com/document/d/1fxmolCv74px-cliKbnmRKSw0j8yovxcLKNcEEg5EOIw/edit
     *
     * Builds and appends the userAdInteractionExt to the provided input URL
     * if the custom event reporting config is enabled
     */
    suspend fun userAdInteractionExtAsQueryParameter(
        eventTimestamp: Long,
        interaction: UserInteraction,
        url: String,
    ): String
}
