package com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport

import android.content.Context
import androidx.annotation.VisibleForTesting
import androidx.work.BackoffPolicy
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.Data
import androidx.work.ListenableWorker
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkRequest
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.android_context.ApplicationContext
import com.moloco.sdk.service_locator.SdkObjectFactory
import java.util.concurrent.TimeUnit

interface PersistentHttpRequest : HttpRequestClient {
    companion object {
        internal fun create(worker: PersistentWorker): PersistentHttpRequest = PersistentHttpRequestImpl(worker)
    }
}

private class PersistentHttpRequestImpl(private val worker: PersistentWorker) :
    PersistentHttpRequest {

    override fun send(url: String) {
        if (!url.isValidUrl()) return
        val data = url.getAsData() ?: return

        MolocoLogger.info(TAG, "Enqueuing request to $url")
        val workRequest = createWorkRequest<UrlGetRequestWorker>(data)

        worker.enqueue(workRequest)
    }

    private inline fun <reified W : ListenableWorker> createWorkRequest(data: Data) =
        OneTimeWorkRequestBuilder<W>()
            .setConstraints(
                Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()
            )
            .setBackoffCriteria(
                BackoffPolicy.LINEAR,
                WorkRequest.MIN_BACKOFF_MILLIS,
                TimeUnit.MILLISECONDS
            )
            .setInputData(data)
            .build()

    companion object {
        const val TAG = "PersistentHttpRequest"
    }
}

class UrlGetRequestWorker(context: Context, params: WorkerParameters) :
    CoroutineWorker(ApplicationContext(context), params) {
    override suspend fun doWork(): Result {
        val httpClient = SdkObjectFactory.Network.httpClientSingleton

        val url: String by inputData.keyValueMap
        MolocoLogger.info(PersistentHttpRequestImpl.TAG, "Sending request to $url")
        val success = httpClient.sendGetRequest(url)

        return when {
            success -> Result.success()
            runAttemptCount >= WORKER_MAXIMUM_ATTEMPT -> Result.failure()
            else -> Result.retry()
        }
    }
}

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
fun String.getAsData(): Data? {
    // Maximum size is 10KB. So we do a try catch to be sure...
    return try {
        val key = "url"
        val value = this

        workDataOf(key to value)
    } catch (e: Exception) {
        MolocoLogger.error("PersistentHttpRequest", "${e.message}. Url: $this")
        return null
    }
}
