package com.moloco.sdk.internal

import android.view.View
import androidx.annotation.UiThread
import androidx.compose.ui.platform.ComposeView
import androidx.core.view.doOnAttach
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LifecycleRegistry
import androidx.lifecycle.findViewTreeLifecycleOwner
import androidx.lifecycle.setViewTreeLifecycleOwner
import androidx.savedstate.SavedStateRegistry
import androidx.savedstate.SavedStateRegistryController
import androidx.savedstate.SavedStateRegistryOwner
import androidx.savedstate.findViewTreeSavedStateRegistryOwner
import androidx.savedstate.setViewTreeSavedStateRegistryOwner
import com.moloco.sdk.internal.scheduling.runOnMainDispatcher

/**
 * [addLifecycleOwnerSupportTo] **must** be called on the wrapping container view.
 *
 * If it is not called on the wrapping container, the call will be too late, resulting in a
 * `ViewTreeLifecycleOwner not found` exception. This happens because [ComposeView] immediately checks
 * for missing essential components in its `WindowAttached` event handler.
 *
 * When [addLifecycleOwnerSupportTo] is called on the wrapping container,
 * the function injects the required components (if missing) during the wrapping container's
 * `window attach` event. This is the point where it is possible to access the `rootView` and
 * apply the fix properly.
 *
 * Since the wrapping container's `window attach` event occurs before the inner [ComposeView]'s
 * `window attach` event, the patch is applied just in time. This ensures that when the
 * [ComposeView] checks for missing components during its `window attach` event, the required
 * components are already present, allowing the process to continue without issues.
 */
internal fun interface ViewLifecycleOwner {
    @UiThread
    fun addLifecycleOwnerSupportTo(view: View)
}

/**
 * A class that provides `LifecycleOwner` and `SavedStateRegistryOwner` implementations
 * for Jetpack Compose-based views in non-appcompat activities.
 *
 * This class helps to manage lifecycle events and saved state restoration for views
 * that need to operate in environments where the typical appcompat libraries aren't present.
 */
internal class ViewLifecycleOwnerImpl : ViewLifecycleOwner, LifecycleOwner, SavedStateRegistryOwner {
    override val lifecycle: Lifecycle get() = lifecycleRegistry
    override val savedStateRegistry: SavedStateRegistry get() = savedStateRegistryController.savedStateRegistry
    private val lifecycleRegistry = LifecycleRegistry(this)
    private val savedStateRegistryController = SavedStateRegistryController.create(this)

    /**
     * *Use this workaround function to make Jetpack Compose based views work in [android.app.Activity] based activities.*
     *
     * *Call it on [android.view.View] which extends and/or contains Jetpack Compose views PRIOR WindowAttached event: e.g. view ctor or right before/after adding view to the container.*
     *
     * *Compat based activities do not need this workaround.*
     *
     * Logic:
     *
     * Whenever this [android.view.View] is attached to window: set absent ViewTreeLifecycleOwner and SavedStateRegistry for the non-appcompat/vanilla activity in order to support
     * Jetpack Compose views in such activities, since Jetpack Compose view crashes the app without those:
     *
     * `java.lang.IllegalStateException: ViewTreeLifecycleOwner not found from xxxView(..`
     *
     */
    override fun addLifecycleOwnerSupportTo(view: View) = with(view) {
        runOnMainDispatcher {
            doOnAttach {
                trySetViewTreeLifecycleOwnerAndSavedStateRegistryController(this)
            }
        }
        Unit
    }

    /*
     * Sets absent ViewTreeLifecycleOwner and SavedStateRegistry for the non-appcompat/vanilla activity in order to support
     * Jetpack Compose views in such activities, since Jetpack Compose view crashes the app without those:
     * java.lang.IllegalStateException: ViewTreeLifecycleOwner not found from xxxView(..
    */
    private fun trySetViewTreeLifecycleOwnerAndSavedStateRegistryController(view: View): Unit = with(view.rootView) {
        // Safety check, as some versions of Android and also some wrongly configured Android devices might return null
        if (this == null) return@with

        /**
         * Only set owner if absent
         */
        if (findViewTreeSavedStateRegistryOwner() == null) {
            setViewTreeSavedStateRegistryOwner(this@ViewLifecycleOwnerImpl)
            savedStateRegistryController.performRestore(null)
            MolocoLogger.info(TAG, "ViewTreeSavedStateRegistryOwner is absent, setting custom one")
        }

        /**
         * Only set owner if absent
         */
        if (findViewTreeLifecycleOwner() == null) {
            setViewTreeLifecycleOwner(this@ViewLifecycleOwnerImpl)
            lifecycleRegistry.handleLifecycleEvent(Lifecycle.Event.ON_CREATE)
            lifecycleRegistry.handleLifecycleEvent(Lifecycle.Event.ON_START)
            lifecycleRegistry.handleLifecycleEvent(Lifecycle.Event.ON_RESUME)
            MolocoLogger.info(TAG, "ViewTreeLifecycleOwner is absent, setting custom one")
        }
    }

    companion object {
        private const val TAG = "ViewLifecycleOwner"
    }
}
