package com.moloco.sdk.internal.publisher

import android.annotation.SuppressLint
import android.content.Context
import android.view.Gravity
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.BannerSize
import com.moloco.sdk.internal.DefaultMolocoBannerAggregatedOptions
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.ViewLifecycleOwner
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.createInternalAdErrorInfo
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.Player
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.internal.toBannerAggregatedOptions
import com.moloco.sdk.internal.toPx
import com.moloco.sdk.publisher.*
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.Banner
import com.moloco.sdk.publisher.MolocoAdError.ErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.*
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import com.moloco.sdk.acm.AndroidClientMetrics as acm

internal typealias XenossBanner<L> = com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Banner<L>

internal fun Banner(
        context: Context,
        appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
        customUserEventBuilderService: CustomUserEventBuilderService,
        adUnitId: String,
        verifyBannerVisible: Boolean,
        externalLinkHandler: ExternalLinkHandler,
        watermark: Watermark,
        adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
        viewLifecycleOwnerSingleton: ViewLifecycleOwner,
        bannerSize: BannerSize
): Banner = BannerImpl(
    context,
    appLifecycleTrackerService,
    customUserEventBuilderService,
    adUnitId,
    verifyBannerVisible,
    externalLinkHandler,
    ::createAggregatedBanner,
    ::createAggregatedAdShowListener,
    watermark,
    adCreateLoadTimeoutManager,
    viewLifecycleOwnerSingleton,
    bannerSize
)

private class InternalBannerAdShowListenerTracker(
    val originListener: BannerAdShowListener?,
    appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    customUserEventBuilderService: CustomUserEventBuilderService,
    provideSdkEvents: () -> SdkEvents?,
    provideBUrlData: () -> BUrlData?,
    adType: AdFormatType
) : InternalAdShowListener by InternalAdShowListenerTracker(
    originListener,
    appLifecycleTrackerService,
    customUserEventBuilderService,
    provideSdkEvents,
    provideBUrlData,
    adType = adType
)

private class BannerAdDataHolder<L : VastAdShowListener>(
    var xenossBanner: XenossBanner<L>? = null,
    var sdkEvents: SdkEvents? = null,
    var bUrlData: BUrlData? = null,
    var adDisplayStateJob: Job? = null,
)

// TODO. Refactor Duplicate code (FullscreenAd)
@SuppressLint("ViewConstructor")
internal class BannerImpl<L : VastAdShowListener>(
    private val context: Context,
    private val appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val adUnitId: String,
    private val verifyBannerVisible: Boolean,
    private val externalLinkHandler: ExternalLinkHandler,
    private val createXenossBanner: (
        context: Context,
        customUserEventBuilderService: CustomUserEventBuilderService,
        bid: Bid,
        externalLinkHandler: ExternalLinkHandler,
        watermark: Watermark
    ) -> XenossBanner<L>,
    createXenossBannerAdShowListener: (basedOnThisVastAdShowListener: VastAdShowListener) -> L,
    private val watermark: Watermark,
    private val adCreateLoadTimeoutManager:AdCreateLoadTimeoutManager,
    viewLifecycleOwner:ViewLifecycleOwner,
    private val bannerSize: BannerSize
) : Banner(context), CreateAdObjectTime by adCreateLoadTimeoutManager {
    init {
        viewLifecycleOwner.addLifecycleOwnerSupportTo(this)
    }

    private val bannerCreateToLoadTimerEvent: TimerEvent = acm.startTimerEvent(AcmTimer.CreateToLoad.eventName).withTag(AcmTag.AdType.tagName, AdFormatType.BANNER.name.lowercase())
    private var loadToShowTimerEvent: TimerEvent? = null
    private val scope = CoroutineScope(DispatcherProvider().main)

    private val adDataHolder = BannerAdDataHolder<L>()

    // Internal ad show listener tracker to track ad show events including SDK internal error events
    private var internalAdShowListener: InternalBannerAdShowListenerTracker? = null

    // NOTE - Never use `adShowListener` directly in this class, use [internalAdShowListener]
    override var adShowListener: BannerAdShowListener? = null
        set(value) {
            val bannerListener = createInternalBannerAdShowListenerTracker(value)
            internalAdShowListener = bannerListener
            field = bannerListener.originListener
        }

    private fun createInternalBannerAdShowListenerTracker(
        originListener: BannerAdShowListener?
    ): InternalBannerAdShowListenerTracker = InternalBannerAdShowListenerTracker(
        originListener = originListener,
        appLifecycleTrackerService = appLifecycleTrackerService,
        customUserEventBuilderService = customUserEventBuilderService,
        provideSdkEvents = { adDataHolder.sdkEvents },
        provideBUrlData = { adDataHolder.bUrlData },
        adType = AdFormatType.BANNER,
    )

    private fun destroyAd(sendErrorEvent: MolocoInternalAdError? = null) {
        with(adDataHolder) {
            adDisplayStateJob?.cancel()
            adDisplayStateJob = null
        }

        // we need to create variable here, before destroying `xenossBanner` object
        val isAdShowing = isAdShowing(adDataHolder.xenossBanner).value
        with(adDataHolder) {
            xenossBanner?.destroy()
            xenossBanner = null
        }

        sendErrorEvent?.let {
            // TODO: https://mlc.atlassian.net/browse/SDK-1730
            // TODO: Pass the right error type
            internalAdShowListener?.onAdShowFailed(it)
        }
        // Make sure ad hidden event is sent event when the ad is destroyed during ad display.
        if (isAdShowing) internalAdShowListener?.onAdHidden(createAdInfo(adUnitId))

        // TODO.
        //  Quick workaround for when sdkEvents become null too early in FullscreenAd or Banner due to destroyAd() call,
        //  which leads to onAdHidden, onError not track events.
        adDataHolder.sdkEvents = null

        adDataHolder.bUrlData = null
    }

    override fun destroy() {
        scope.cancel()
        destroyAd()
        adShowListener = null
        internalAdShowListener = null
    }

    private val adLoader = AdLoad(
            scope,
            adCreateLoadTimeoutManager::calculateTimeout,
            adUnitId,
            ::recreateXenossAd,
            adFormatType = AdFormatType.BANNER)

    private fun recreateXenossAd(
        bid: Bid,
    ): com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad {
        destroyAd()

        val xenossBanner = createXenossBanner(context, customUserEventBuilderService, bid, externalLinkHandler, watermark)

        with(adDataHolder) {
            this.xenossBanner = xenossBanner
            sdkEvents = bid.ext?.sdkEvents
            bUrlData = if (bid.burl != null) BUrlData(bid.burl, bid.price) else null
        }

        with(xenossBanner) {
            adShowListener = xenossBannerAdShowListener
            listenToAdDisplayState()
        }

        val layoutParams = LayoutParams(
            bannerSize.wDp.toPx(),
            bannerSize.hDp.toPx()
        ).apply {
            gravity = Gravity.CENTER
        }

        addView(xenossBanner, layoutParams)

        return xenossBanner
    }

    override val isLoaded: Boolean
        get() = adLoader.isLoaded

    override fun load(bidResponseJson: String, listener: AdLoad.Listener?) {
        // We want the createToLoad timer as close to the public API calls as possible.
        acm.recordTimerEvent(bannerCreateToLoadTimerEvent)
        loadToShowTimerEvent = acm.startTimerEvent(AcmTimer.LoadToShow.eventName)
        // Wrapping by launch { ... } block to make sure the fun call is executed on the main thread.
        scope.launch {
            adLoader.load(bidResponseJson, listener)
        }
    }

    /**
     * Returns a [StateFlow] that emits a boolean value indicating whether the ad is currently
     * showing. If the [verifyBannerVisible] parameter is `true` or the [xenossBanner] is `null`,
     * this function returns a [StateFlow] that emits the value of [banner.isViewShown]. If
     * the [verifyBannerVisible] parameter is `false` this function returns a [StateFlow]
     * that emits the value of [xenossBanner.isAdDisplaying].
     *
     * @return a [StateFlow] that emits a boolean value indicating whether the ad is currently showing.
     */
    private fun isAdShowing(xenossBanner: XenossBanner<L>?): StateFlow<Boolean> {
        xenossBanner.apply {
            return if (verifyBannerVisible || this == null) {
                isViewShown
            } else {
                isAdDisplaying
            }
        }
    }

    private fun AdDisplayState.listenToAdDisplayState() {
        with(adDataHolder) {
            adDisplayStateJob?.cancel()
            adDisplayStateJob = isAdShowing(adDataHolder.xenossBanner)
                // Ignoring FIRST isAdShowing == false emit, which prevents from firing onAdHidden() event in case
                // if banner is not loaded and/or added to the window yet.
                .dropWhile {
                    !it
                }.onEach {
                    if (it) {
                        loadToShowTimerEvent?.let { event ->
                            acm.recordTimerEvent(event.withTag(AcmTag.AdType.tagName, AdFormatType.BANNER.name.lowercase()))
                        }
                        internalAdShowListener?.onAdShowSuccess(createAdInfo(adUnitId))
                    } else {
                        internalAdShowListener?.onAdHidden(createAdInfo(adUnitId))
                        // Since a natural event sequence is "onAdShowSuccess" followed by "onAdHidden",
                        // according to https://mlc.atlassian.net/browse/SDK-292, we only
                        // need the aforementioned events to fire only once per a successful ad load() call.
                        // Therefore, let's cancel the event listening task after we got 2 events (show and then hidden).
                        adDisplayStateJob?.cancel()
                    }
                }.launchIn(scope)
        }
    }

    private val xenossBannerAdShowListener =
        createXenossBannerAdShowListener(
            object : VastAdShowListener {

                override fun onVastCompletionStatus(skipped: Boolean) {
                    // no-op
                }

                override fun onShowError(internalShowError: MolocoAdSubErrorType) {
                    destroyAd(
                        sendErrorEvent = createInternalAdErrorInfo(adUnitId, ErrorType.AD_SHOW_ERROR, internalShowError)
                    )
                }

                override fun onClick() {
                    internalAdShowListener?.onAdClicked(createAdInfo(adUnitId))
                }
            }
        )

}

private fun generateAggregatedOptions(playerExt: Player?): AggregatedOptions =
    playerExt?.toBannerAggregatedOptions() ?: DefaultMolocoBannerAggregatedOptions()

private fun createAggregatedBanner(
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    bid: Bid,
    externalLinkHandler: ExternalLinkHandler,
    watermark: Watermark
): XenossBanner<AggregatedAdShowListener> = AggregatedBanner(
    context = context,
    customUserEventBuilderService = customUserEventBuilderService,
    bid = bid,
    options = generateAggregatedOptions(bid.ext?.player),
    externalLinkHandler = externalLinkHandler,
    watermark = watermark
)

private fun createAggregatedAdShowListener(basedOnThisVastAdShowListener: VastAdShowListener) =
    object : AggregatedAdShowListener, VastAdShowListener by basedOnThisVastAdShowListener {
    }
