package com.moloco.sdk.internal.publisher.nativead.parser

import android.content.Context
import androidx.core.net.toUri
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.publisher.nativead.NativeAdImpl
import com.moloco.sdk.internal.publisher.nativead.model.NativeOrtbResponse
import com.moloco.sdk.internal.publisher.nativead.model.PreparedNativeAsset
import com.moloco.sdk.internal.publisher.nativead.model.PreparedNativeAssets
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.NativeAdLoadError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastAdLoader
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.coroutineScope
import kotlin.time.Duration
import kotlin.time.DurationUnit
import kotlin.time.toDuration

private const val TAG = "PrepareNativeAssets"
internal suspend fun prepareNativeAssets(
    context: Context,
    assets: List<NativeOrtbResponse.Asset>,
    timeout: Duration,
): Result<PreparedNativeAssets, PrepareNativeAssetException> {
    val loadVast = lazy { VastAdLoader(context) }

    val groupedAssets = assets.groupBy { it.required }
    val requiredAssetsGroup = groupedAssets[true] ?: listOf()
    val optionalAssetsGroup = groupedAssets[false] ?: listOf()

    // Loading required assets firsts, fail everything immediately if any required asset doesn't get "prepared".
    val preparedRequiredAssets = try {
        coroutineScope {
            requiredAssetsGroup.map { asset ->
                async {
                    when (val result = prepareNativeAsset(asset, loadVast, timeout)) {
                        is Result.Success -> {
                            MolocoLogger.info(TAG, "Successfully prepared native asset: ${asset.id}")
                            asset to result
                        }
                        is Result.Failure -> {
                            MolocoLogger.warn(TAG, "Failed to prepare required native asset: ${asset.id}")
                            throw PrepareNativeAssetException(asset.id, result.value)
                        }
                    }
                }
            }.awaitAll()
        }
    } catch (e: PrepareNativeAssetException) {
        MolocoLogger.error(TAG, "Failed to prepare required assets", e)
        return Result.Failure(e)
    }

    val preparedOptionalAssets = coroutineScope {
        optionalAssetsGroup.map { asset ->
            async { asset to prepareNativeAsset(asset, loadVast, timeout) }
        }.awaitAll()
    }

    val data = mutableMapOf<Int, PreparedNativeAsset.Data>()
    val images = mutableMapOf<Int, PreparedNativeAsset.Image>()
    val titles = mutableMapOf<Int, PreparedNativeAsset.Title>()
    val videos = mutableMapOf<Int, PreparedNativeAsset.Video>()
    val failedAssets = mutableListOf<Pair<NativeOrtbResponse.Asset, MolocoAdSubErrorType>>()

    for ((originAsset, preparedNativeAssetResult) in preparedRequiredAssets + preparedOptionalAssets) {
        when (preparedNativeAssetResult) {
            is Result.Failure -> failedAssets += originAsset to preparedNativeAssetResult.value
            is Result.Success -> when (val res = preparedNativeAssetResult.value) {
                is PreparedNativeAsset.Data -> data += res.originAsset.id to res
                is PreparedNativeAsset.Image -> images += res.originAsset.id to res
                is PreparedNativeAsset.Title -> titles += res.originAsset.id to res
                is PreparedNativeAsset.Video -> videos += res.originAsset.id to res
            }
        }
    }

    return Result.Success(PreparedNativeAssets(data, images, titles, videos, failedAssets))
}

private suspend fun prepareNativeAsset(
    asset: NativeOrtbResponse.Asset,
    loadVast: Lazy<VastAdLoader>,
    timeout: Duration
): Result<PreparedNativeAsset, MolocoAdSubErrorType> = when (asset) {
    is NativeOrtbResponse.Asset.Data -> Result.Success(
        PreparedNativeAsset.Data(asset)
    )

    is NativeOrtbResponse.Asset.Image -> prepareImageAsset(
        asset
    )

    is NativeOrtbResponse.Asset.Title -> Result.Success(
        PreparedNativeAsset.Title(asset)
    )

    is NativeOrtbResponse.Asset.Video -> prepareVideoAsset(
        asset,
        loadVast.value,
        timeout,
    )
}

private suspend fun prepareImageAsset(
    asset: NativeOrtbResponse.Asset.Image,
    mediaCacheRepository: MediaCacheRepository = SdkObjectFactory.Media.mediaCacheRepository,
): Result<PreparedNativeAsset, MolocoAdSubErrorType> {
    return when (val mediaCacheResult = mediaCacheRepository.getMediaFile(asset.url)) {
        is MediaCacheRepository.Result.Success -> {
            val imageUri = try {
                MolocoLogger.info(TAG, "Successfully loaded image asset media")
                // Try..catch condition here is not needed, but is a safety net,
                // when the unthinkable happens on an Android device
                mediaCacheResult.file.absolutePath.toUri()
            } catch (e: Exception) {
                // For exceptions, log it as a UNKNOWN error so we can differentiate it
                // from media fetch error
                MolocoLogger.warn(TAG, "Failed to prepare image asset", e)
                return Result.Failure(NativeAdLoadError.NATIVE_AD_IMAGE_PREPARE_ASSET_UNKNOWN_ERROR)
            }

            Result.Success(
                PreparedNativeAsset.Image(
                    asset, imageUri
                )
            )
        }

        else -> {
            MolocoLogger.warn(TAG, "Failed to fetch image asset media")
            Result.Failure(NativeAdLoadError.NATIVE_AD_IMAGE_ASSET_MEDIA_FETCH_ERROR)
        }
    }
}

// TODO. VastAdLoad.kt partial duplication.
private suspend fun prepareVideoAsset(
    asset: NativeOrtbResponse.Asset.Video,
    loadVast: VastAdLoader,
    timeout: Duration,
): Result<PreparedNativeAsset, MolocoAdSubErrorType> {
    // Prepares the VAST ad (Does not actually trigger downloading the media files).
    when (val prepareVastAdResult = loadVast.invoke(adm = asset.vastTag, isStreamingEnabled = true)) {
        is Result.Success -> {
            /**
             * The general timeout for preparing NativeAd assets is defined in [NativeAdImpl.load].
             * Here, we reduce it by 10% to account for streaming overhead, ensuring we stay within the top-level timeout limit.
             */
            val timeToStreamVideo = (timeout.inWholeMilliseconds * 0.9).toDuration(DurationUnit.MILLISECONDS)
            return when (val vastAdResult = loadVast.waitForAdLoadToStart(prepareVastAdResult.value, timeToStreamVideo)) {
                is Result.Success -> {
                    MolocoLogger.info(TAG, "Successfully loaded video asset media")
                    Result.Success(PreparedNativeAsset.Video(asset, vastAdResult.value))
                }

                is Result.Failure -> {
                    MolocoLogger.warn(TAG, "Failed to fetch video asset media: ${vastAdResult.value}")
                    Result.Failure(NativeAdLoadError.NATIVE_AD_VIDEO_ASSET_MEDIA_NOT_ENOUGH_ERROR)
                }
            }
        }

        is Result.Failure -> {
            MolocoLogger.warn(TAG, "Failed to fetch video asset media: ${prepareVastAdResult.value}")
            return Result.Failure(NativeAdLoadError.NATIVE_AD_VIDEO_ASSET_MEDIA_FETCH_ERROR)
        }
    }
}

/**
 * Internal exception when preparing a native ad asset fails.
 */
internal class PrepareNativeAssetException(val assetId: Int, val errorSubType: MolocoAdSubErrorType) : Exception()

