package com.moloco.sdk.internal.services

import android.content.Context
import android.media.AudioManager
import android.os.Build
import androidx.core.content.ContextCompat
import com.moloco.sdk.internal.MolocoLogger

internal enum class MuteSwitchState {
    SILENT,
    VIBRATE,
    NORMAL,
}

internal interface AudioService {
    val isMediaVolumeOnMute: Boolean

    /**
     * Get the mute switch state of the device
     */
    val muteSwitchState: MuteSwitchState

    /**
     * Get the volume of audio streams for music playback
     */
    val mediaVolume: Int
}

internal class AudioServiceImpl(
    context: Context
) : AudioService {

    private val audioManager: AudioManager? = try {
        ContextCompat.getSystemService(
            context.applicationContext,
            AudioManager::class.java
        )
    } catch (e: Exception) {
        MolocoLogger.error(TAG, "couldn't get android.media.AudioManager service", e)
        null
    }

    override val isMediaVolumeOnMute: Boolean
        get() =
            // In case audio manager is unavailable, fall back to "mute" value.
            if (audioManager == null) {
                true
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                audioManager.isStreamMute(StreamMusic)
            } else {
                audioManager.getStreamVolume(StreamMusic) == 0
            }

    override val muteSwitchState: MuteSwitchState
        get() = when (audioManager?.ringerMode) {
            AudioManager.RINGER_MODE_SILENT -> MuteSwitchState.SILENT
            AudioManager.RINGER_MODE_VIBRATE -> MuteSwitchState.VIBRATE
            AudioManager.RINGER_MODE_NORMAL -> MuteSwitchState.NORMAL
            else -> MuteSwitchState.NORMAL
        }

    override val mediaVolume: Int
        get() = audioManager?.getStreamVolume(AudioManager.STREAM_MUSIC) ?: 0
}

private const val TAG = "AudioService"

private const val StreamMusic = AudioManager.STREAM_MUSIC
