package com.moloco.sdk.internal.services.analytics

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.GlobalScopes
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventConfigService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import kotlinx.coroutines.launch

/**
 * Analytics service that can record analytics events and upload them
 */
interface AnalyticsService {
    fun recordApplicationForeground(eventTimestamp: Long, bgEventTimestamp: Long)
    fun recordApplicationBackground(eventTimestamp: Long)
}

private const val TAG = "AnalyticsService"
internal class AnalyticsServiceImpl(
    private val persistentHttpRequest: PersistentHttpRequest,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val configService: CustomUserEventConfigService
) : AnalyticsService {
    override fun recordApplicationForeground(eventTimestamp: Long, bgEventTimestamp: Long) {
        if (configService.isCustomUserEventReportingEnabled() && configService.applicationForegroundUrl().isNotEmpty()) {
            MolocoLogger.debug(
                TAG,
                "Recording applicationForeground with timestamp: $eventTimestamp, lastBgTimestamp: $bgEventTimestamp"
            )
            GlobalScopes.globalIOScope.launch {
                val url = customUserEventBuilderService.userAdInteractionExtAsQueryParameter(
                    eventTimestamp,
                    CustomUserEventBuilderService.UserInteraction.AppForeground(bgEventTimestamp),
                    configService.applicationForegroundUrl()
                )
                persistentHttpRequest.send(url)
            }
        }
    }

    override fun recordApplicationBackground(eventTimestamp: Long) {
        if (configService.isCustomUserEventReportingEnabled() && configService.applicationBackgroundUrl().isNotEmpty()) {
            MolocoLogger.debug(
                TAG,
                "Recording applicationBackground with timestamp: $eventTimestamp"
            )
            GlobalScopes.globalIOScope.launch {
                val url = customUserEventBuilderService.userAdInteractionExtAsQueryParameter(
                    eventTimestamp,
                    CustomUserEventBuilderService.UserInteraction.AppBackground,
                    configService.applicationBackgroundUrl()
                )
                persistentHttpRequest.send(url)
            }
        }
    }
}
