package com.moloco.sdk.internal.services.bidtoken

import android.util.Base64
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.MolocoLogger.debugBuildLog
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.services.bidtoken.providers.ClientBidTokenSignalProvider
import com.moloco.sdk.internal.services.bidtoken.providers.CompositeClientBidTokenSignalProvider
import com.moloco.sdk.internal.services.bidtoken.providers.CompositeClientBidTokenSignalProviderImpl
import com.moloco.sdk.internal.services.bidtoken.providers.MemorySignalProvider
import com.moloco.sdk.internal.services.bidtoken.providers.PrivacyStateSignalProvider
import com.moloco.sdk.internal.services.bidtoken.providers.SDKInitStateSignalProvider
import com.moloco.sdk.internal.services.encryption.EncryptionService
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.service_locator.SdkObjectFactory
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.security.InvalidAlgorithmParameterException
import java.security.InvalidKeyException
import java.security.NoSuchAlgorithmException
import java.security.spec.InvalidKeySpecException
import javax.crypto.BadPaddingException
import javax.crypto.IllegalBlockSizeException
import javax.crypto.NoSuchPaddingException

private const val CLIENT_BID_TOKEN_VERSION = "v2"

/**
 * Client side bid token provider as per the proposal here:
 * https://docs.google.com/document/d/1elJPg3JNxJ4s6qo1OdHDlo6mOgTNtBOHwctkJDl5r4o/edit#heading=h.rvg2y6iz8zo9
 */
internal interface ClientBidTokenService {
    companion object {
        fun create(): ClientBidTokenService =
            ClientBidTokenServiceImpl(
                timeProviderService = SdkObjectFactory.Miscellaneous.timeProviderSingleton,
                clientBidTokenBuilder = ClientBidTokenBuilder.create(),
                encryptionService = EncryptionService.create(),
                signalProvider = CompositeClientBidTokenSignalProvider.create(),
            )
    }

    suspend fun bidToken(publicKey: String, bidTokenConfig: BidTokenConfig): Result<String>
}

internal class ClientBidTokenServiceImpl(
    private val timeProviderService: TimeProviderService,
    private val clientBidTokenBuilder: ClientBidTokenBuilder,
    private val encryptionService: EncryptionService,
    private val signalProvider: CompositeClientBidTokenSignalProvider,
) : ClientBidTokenService {
    private val TAG = "ClientBidTokenServiceImpl"
    private var rsaPublicKey: String = ""
    private var clientBidToken: String = ""

    private var config: BidTokenConfig = DefaultBidTokenConfig

    private val mutex = Mutex()

    override suspend fun bidToken(publicKey: String, bidTokenConfig: BidTokenConfig): Result<String> {
        mutex.withLock {
            if (shouldRefreshClientBidToken(publicKey, bidTokenConfig)) {
                debugBuildLog(TAG, "Bid token needs refresh, fetching new bid token")
                rsaPublicKey = publicKey
                config = bidTokenConfig
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenCached.eventName).withTag(AcmTag.Result.tagName, "false"))
                clientBidToken = clientBidToken(publicKey)
            } else {
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenCached.eventName).withTag(AcmTag.Result.tagName, "true"))
            }

            if (clientBidToken.isEmpty()) {
                return Result.failure(Exception("Client bid token is empty"))
            }

            return Result.success(clientBidToken)
        }
    }

    private fun tryUpdateConfig(newConfig: BidTokenConfig): Boolean {
        val oldConfig = config
        config = newConfig

        return (oldConfig != newConfig).also {
            debugBuildLog(TAG, if (it) "config updated" else "config didn't change")
        }
    }

    private fun shouldRefreshClientBidToken(publicKey: String, config: BidTokenConfig): Boolean {
        val rsaPublicKeyUpdated = (rsaPublicKey != publicKey)
        if (rsaPublicKeyUpdated) {
            debugBuildLog(TAG, "rp changed, needs refresh")
            return true
        }

        val configUpdated = tryUpdateConfig(config)
        if (configUpdated) {
            debugBuildLog(TAG, "config changed, needs refresh")
            return true
        }

        if (clientBidToken.isEmpty()) {
            debugBuildLog( TAG, "cached bidToken is empty, needs refresh")
            return true
        }


        if (signalProvider.needsRefresh()) {
            debugBuildLog(TAG, "signal provider updated, needs refresh")
            return true
        }


        debugBuildLog(TAG, "Bid token doesn't need refresh")
        return false
    }

    private fun clientBidToken(publicKey: String): String {
        if (publicKey.isEmpty()) {
            AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenBuild.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag(AcmTag.Reason.tagName, "empty_public_key"))
            return ""
        }

        val buildTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.ClientBidTokenBuild.eventName)
        val failureReason: String
        try {
            val startTime = timeProviderService.currentTime()
            val encryptedAesKey = encryptionService.rsaEncryptedAesKey(publicKey)
            // Before building a new client bidtoken, always update all the signal providers
            signalProvider.tryUpdateSignalState()
            // Get the signal from the signal provider needed for client bid token
            val clientSignals = signalProvider.provideSignal()
            // Build the client bid token proto
            val clientBidTokenComponent = clientBidTokenBuilder.buildClientBidTokenComponent(clientSignals, config)
            // Base64 encode the client bid token component
            val base64clientBidTokenComponent = Base64.encode(clientBidTokenComponent.toByteArray(), Base64.DEFAULT)
            // AES encrypt the base64 encoded client bid token component
            val encryptedClientBidToken = encryptionService.aesEncrypt(base64clientBidTokenComponent)
            // Base64 encode the encrypted client bid token component
            val base64BidToken = Base64.encode(encryptedClientBidToken, Base64.DEFAULT)
            // Build the client bid token
            val clientBidToken = clientBidTokenBuilder.buildBidToken(base64BidToken, encryptedAesKey)
            // Base64 encode the client bid token
            val base64EncodedClientBidTokenPayload = Base64.encodeToString(clientBidToken, Base64.DEFAULT)
            
            AndroidClientMetrics.recordTimerEvent(buildTimerEvent.withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenBuild.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            MolocoLogger.info(TAG, "Client bid token build time: ${timeProviderService.currentTime() - startTime} ms")
            return "$CLIENT_BID_TOKEN_VERSION:$base64EncodedClientBidTokenPayload"
        } catch (nsae: NoSuchAlgorithmException) {
            failureReason = "no_such_algorithm_exception"
        } catch (nspe: NoSuchPaddingException) {
            failureReason = "no_such_padding_exception"
        } catch (ike: InvalidKeyException) {
            failureReason = "invalid_key_exception"
        } catch (ibse: IllegalBlockSizeException) {
            failureReason = "illegal_block_size_exception"
        } catch(bpe: BadPaddingException) {
            failureReason = "bad_padding_exception"
        } catch(iape: InvalidAlgorithmParameterException) {
            failureReason = "invalid_algorithm_parameter_exception"
        } catch(ikse: InvalidKeySpecException) {
            failureReason = "invalid_key_spec_exception"
        } catch (iae: IllegalArgumentException) {
            failureReason = "illegal_argument_exception"
        } catch (e: Exception) {
            failureReason = "unknown_exception"
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenBuild.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag(AcmTag.Reason.tagName, failureReason))
        AndroidClientMetrics.recordTimerEvent(buildTimerEvent.withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag(AcmTag.Reason.tagName, failureReason))
        return ""
    }
}
