package com.moloco.sdk.publisher

/**
 * Data class representing a Moloco ad error that is visible for the PUBLIC API.
 *
 * @property networkName The name of the ad network.
 * @property adUnitId The id of the ad unit.
 * @property errorType The type of the error.
 * @property description The description of the error (default: the description of the [errorType]).
 */
data class MolocoAdError(
    val networkName: String,
    val adUnitId: String,
    val errorType: ErrorType,
    val description: String = errorType.description,
) {

    /**
     * Enum representing the types of errors that can occur when loading or displaying a Moloco ad.
     *
     * Link to the document describing the error codes:
     * https://www.notion.so/moloco/Tech-Design-MAX-Mediation-Integration-Proposal-bf75f57e20274941aae96cd3aff9e07f#f6797faafcfd45cc96fe6d435aa03f25
     */
    enum class ErrorType(val description: String, val errorCode: Int) {
        /**
         * Generic failure error code.
         * Error code "-1"
         */
        UNKNOWN("Generic failure error code", -1),

        /**
         * SDK initialization error.
         * Error code "1000"
         */
        SDK_INIT_ERROR("SDK initialization error", 1000),

        /**
         * WorkManager crash that prevents us from creating a persistent http request client.
         * Error code "1001"
         */
        SDK_PERSISTENT_HTTP_REQUEST_FAILED_TO_INIT(
        "WorkManager crash that prevents us from creating a persistent http request client",
        1001
        ),

        /**
         * Ad object cannot be created, due to invalid configuration.
         * Error code "1002"
         */
        SDK_INVALID_CONFIGURATION(
        "Ad object cannot be created, due to invalid configuration",
        1002
        ),

        /**
         * Ad is not loaded.
         * Error code "2000"
         */
        AD_LOAD_FAILED("Ad is not loaded", 2000),

        /**
         * Ad load failed because SDK was not initialized.
         * Error code "2001"
         */
        AD_LOAD_FAILED_SDK_NOT_INIT("Ad load failed because SDK was not initialized", 2001),

        /**
         * Ad failed to load because the request timed out.
         * Error code "2003"
         */
        AD_LOAD_TIMEOUT_ERROR("Ad failed to load because the request timed out", 2003),

        /**
         * Ad load limit reached.
         * Error code "2004"
         */
        AD_LOAD_LIMIT_REACHED("Ad load limit reached", 2004),

        /**
         * Ad failed to load because bid request failed or no fill.
         * Error code "2005"
         */
        AD_LOAD_BID_FAILED("Ad failed to load because bid request failed or no fill", 2005),

        /**
         * Ad show failed despite a successful ad load. Internal renderer error.
         * Error code "3000"
         */
        AD_SHOW_ERROR(
        "Ad show failed despite a successful ad load. Internal renderer error",
        3000
        ),

        /**
         * Ad show failed because ad load was not called.
         * Error code "3001"
         */
        AD_SHOW_ERROR_NOT_LOADED("Ad show failed because ad load was not called", 3001),

        /**
         * Ad show failed because show() was invoked multiple times when ad was already rendering.
         * Error code "3002"
         */
        AD_SHOW_ERROR_ALREADY_DISPLAYING(
        "Ad show failed because show() was invoked multiple times when ad was already rendering",
        3002
        ),

        /**
         * The bid response received was not parseable by the SDK.
         * Error code "4000"
         */
        AD_BID_PARSE_ERROR("The bid response received was not parseable by the SDK", 4000),

        /**
         * The bid token couldn’t be retrieved.
         * Error code "5000"
         */
        AD_SIGNAL_COLLECTION_FAILED("The bid token couldn’t be retrieved.", 5000),
    }

    /**
     * Enum class representing different errors that can occur during the creation of a Moloco ad object.
     *
     * @property description A detailed description of the error.
     * @property errorCode A unique code identifying the error.
     */
    enum class AdCreateError(val description: String, val errorCode: Int) {

        /**
         * Error indicating that the ad unit ID provided could not be found or is invalid. Error code: 1010
         */
        INVALID_AD_UNIT_ID("Could not find the adUnitId that was requested for load", 1010),

        /**
         * Error indicating that the SDK initialization failed, preventing the creation of the ad object. Error code: 1011
         */
        SDK_INIT_FAILED("Failed to create `ad` object due to SDK initialization failure.", 1011),

        /**
         * Error indicating that `Moloco.initialize()` was not called or did not complete, which is required before creating ad objects.  Error code: 1012
         */
        SDK_INIT_WAS_NOT_COMPLETED("Failed to create `ad` object due to `Moloco.initialize()` not complete.", 1012)
    }
}



/**
 * Creates a new [MolocoAdError] object with [MolocoAdError.networkName] set to "Moloco".
 *
 * @param adUnitId The id of the ad unit.
 * @param errorType The type of the error.
 * @return A new [MolocoAdError] object with [MolocoAdError.networkName] set to "Moloco".
 */
internal fun createAdErrorInfo(
    adUnitId: String,
    errorType: MolocoAdError.ErrorType
): MolocoAdError = MolocoAdError("Moloco", adUnitId, errorType)
