package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer

import android.content.Context
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.AggregatedFullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.StateFlow

/**
 * Common public API for all fullscreen ad renderers (mraid, vast, static etc) <br>

 * The use-case scenario is: <br>
 * 1. Create an instance of [FullscreenAd] <br>
 * 2. call [AdLoad.load] <br>
 * 3. wait for [AdLoad.Listener.onLoad] callback <br>
 * 4. call [show] function <br>
 *
 * CLEANING UP RESOURCES:<br>
 *
 * In case of getting [AdDisplayState.isAdDisplaying] == false AFTER displaying the ad, [AdLoad.Listener.onLoadTimeout], [AdLoad.Listener.onLoadError] or [AdShowListener.onShowError] callback events, or simply when [android.app.Activity] is destroyed, in order to prevent memory leaks make sure to call [Destroyable.destroy] method <br>
 *
 * CAVEATS: <br>
 *
 * Current implementations of [FullscreenAd] interface do not support multiple [AdLoad.load] and [show] calls: results may be unpredictable <br>
 */
interface FullscreenAd<T : AdShowListener, R> : CreativeTypeProvider, AdLoad, AdDisplayState, FullscreenAdCloseState, Destroyable {

    // TODO. Suspendable show with return (success|error) or returns event flow + starts displaying.
    /**
     * @param options mostly for changing renderer behaviour, like button customization,
     * when to show close button, mute etc. Same as for [Banner.options]
     */
    fun show(options: R, listener: T?)

    // Todo.
    //  1. Implement isAdLoaded, isDisplaying observable properties.
    //  2. Discuss if we need to  make load() and show() methods suspendable?
    //  3. Consider replacing listener with StateFlow and/or SharedFlow?
}

/**
 * Detects if the fullscreen ad was forcibly closed by the system or of the user
 * Returns true if ad was closed by system and false otherwise
 */
interface FullscreenAdCloseState {
    val isAdForciblyClosed: StateFlow<Boolean>
}

/**
 * Fullscreen ad that resolves creative type (mraid, vast etc) automatically.
 *
 * @param context The context to be used for creating the ad.
 * @param customUserEventBuilderService A service for building custom user events.
 * @param creativeType Optional parameter that bypasses the creative resolver.
 * @param bid The bid associated with the ad.
 * @param externalLinkHandler A handler for managing external links.
 * @param watermark The watermark to be applied to the ad.
 *
 * @return An instance of [FullscreenAd] with [AggregatedAdShowListener] and [AggregatedOptions].
 */
internal fun AggregatedFullscreenAd(
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    creativeType: CreativeType? = null,
    bid: Bid,
    externalLinkHandler: ExternalLinkHandler,
    watermark: Watermark
): FullscreenAd<AggregatedAdShowListener, AggregatedOptions> = AggregatedFullscreenAd(
    context = context,
    customUserEventBuilderService = customUserEventBuilderService,
    creativeType = creativeType,
    bid = bid,
    externalLinkHandler = externalLinkHandler,
    watermark = watermark
)
