package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.annotation.SuppressLint
import android.content.Context
import androidx.compose.ui.unit.dp
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.*
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toStaticBannerError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.StaticWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

/**
 * @param options mostly for changing renderer behaviour, like button customization, when to show close button, mute etc. Same as for [FullscreenAd.show]
 */
internal fun StaticBanner(
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    adm: String,
    options: AdWebViewOptions,
    externalLinkHandler: ExternalLinkHandler,
    watermark: Watermark
): Banner<AdShowListener> = StaticBannerImpl(
    context,
    customUserEventBuilderService,
    adm,
    options,
    externalLinkHandler,
    watermark
)

@SuppressLint("ViewConstructor")
internal class StaticBannerImpl(
    private val context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    adm: String,
    private val options: AdWebViewOptions,
    externalLinkHandler: ExternalLinkHandler,
    private val watermark: Watermark
) : Banner<AdShowListener>(context) {
    init {
        tag = "MolocoStaticBannerView"
    }

    override val creativeType = CreativeType.STATIC

    /**
     * Indicates the padding at which the Ad Badge should be rendered in the bottom left corner (the i icon)
     */
    private val AD_BADGE_PADDING = 5.dp

    private val staticWebView: StaticWebView = StaticWebView(
        context,
        customUserEventBuilderService,
        externalLinkHandler,
    )

    override fun destroy() {
        super.destroy()
        staticWebView.destroy()
    }

    override val adLoader: AdLoad = StaticAdLoad(adm, scope, staticWebView)

    override fun prepareAdViewForDisplay() {
        scope.launch {
            launch {
                val error = staticWebView.unrecoverableError.first { it != null }
                error?.let {
                    adShowListener?.onShowError(it.toStaticBannerError())
                }
            }
            launch {
                staticWebView.clickthroughEvent.collect {
                    adShowListener?.onClick()
                }
            }

            adView = options.AdWebViewRenderer(
                context,
                staticWebView,
                options.closeDelaySeconds,
                MutableStateFlow(false),
                staticWebView::onButtonRendered,
                { /* no-op  */ },
                watermark,
                AD_BADGE_PADDING,
                false
            )
        }
    }

    override val isAdDisplaying: StateFlow<Boolean> by lazy {
        // Added WebView's "isPageFinished" event check for static ads to prevent impression discrepancies,
        // between notifnt and html embedded trackers.
        super.isAdDisplaying.combine(staticWebView.isPageFinished) { superIsAdDisplaying, isPageFinished ->
            superIsAdDisplaying && isPageFinished
        }.stateIn(scope, SharingStarted.Eagerly, false)
    }
}
