package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import android.os.Bundle
import android.view.ViewGroup
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.annotation.VisibleForTesting
import androidx.compose.runtime.Composable
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.android_context.ApplicationContext
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.closeDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.decDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putCloseDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.putDECDelaySeconds
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultAdCloseCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.mraidAdPlaylistController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultAdSkipCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview.AdWebViewRenderer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview.MraidAdContainerScreen
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import java.lang.ref.WeakReference

internal class MraidActivity : ComponentActivity() {

    private val scope = CoroutineScope(DispatcherProvider().main)
    private val customUserEventBuilderService by lazy { SdkObjectFactory.UserTracking.customUserEventBuilderFactory }
    private var _adPlaylistController: AdController? = null
    private var _mraidContentController: MraidFullscreenContentController? = null

    private fun startListeningToMraidOrientationEvent(
        expectedOrientation: StateFlow<MraidJsCommand.SetOrientationProperties?>,
    ) {
        // Speed up things a lil bit ot avoid screen rotation jitter..
        setOrientation(expectedOrientation.value)
        // TODO. What about when user change orientation manually
        //  and mraid gonna send a different orientation? Edge case but retarded...
        expectedOrientation.onEach(::setOrientation).launchIn(scope)
    }

    private fun setOrientation(orientationCommand: MraidJsCommand.SetOrientationProperties?) {
        // TODO. Orientation locking.
        //  Let's ignore it for now.
        // it.allowOrientationChange ... SCREEN_ORIENTATION_LOCKED ...

        orientationCommand
            ?.forceOrientation
            ?.toActivityOrientation()
            ?.let { requestedOrientation = it }
    }

    private fun MraidOrientation.toActivityOrientation() = when (this) {
        MraidOrientation.Portrait -> ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        MraidOrientation.Landscape -> ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        MraidOrientation.None -> null // Nothing...
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        ApplicationContext(applicationContext)

        MraidActivityDataHolder.weakActivity = this

        val closeButton = MraidActivityDataHolder.closeButton
        val mraidRenderer = MraidActivityDataHolder.mraidRenderer
        if (mraidRenderer == null) {
            MolocoLogger.error(TAG, "can't display ad: MraidRenderer is missing")
            finish()
            return
        }

        val mraidContentController = MraidActivityDataHolder.weakController

        if (mraidContentController == null) {
            MolocoLogger.error(TAG, "can't display ad: mraid controller is missing")
            finish()
            return
        }

        val adPlaylistController = MraidActivityDataHolder.adData?.let {
            mraidAdPlaylistController(
                mraidAdData = it,
                externalLinkHandler = SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                context = this,
                mraidFullscreenContentController = mraidContentController,
                decGoNextActionDelaySeconds = intent.decDelaySeconds,
                customUserEventBuilderService = customUserEventBuilderService
            )
        }

        if (adPlaylistController == null) {
            MolocoLogger.error(TAG, "can't display ad: mraid ad data is missing")
            finish()
            return
        }

        startListeningToMraidOrientationEvent(mraidContentController.expectedOrientation)

        adPlaylistController.event.onEach {
            currentMraidEvents.emit(it)
            if (it.isErrorOrDismiss) {
                finish()
            }
        }.launchIn(scope)

        setContent {
            MraidAdContainerScreen(
                adViewModel = adPlaylistController,
                webView = mraidContentController.webView,
                closeDelaySeconds = intent.closeDelaySeconds,
                onButtonRendered = { /* NO OP: MRAID not supported */ },
                onClose = mraidContentController::onSkipOrClose,
                adWebViewRenderer = mraidRenderer,
                watermark = MraidActivityDataHolder.watermark,
                adCloseCountdownButton = closeButton(),
                adSkipCountdownButton = defaultAdSkipCountdownButton()
            )
        }

        adPlaylistController.show()
        _adPlaylistController = adPlaylistController
        _mraidContentController = mraidContentController
    }

    override fun onDestroy() {
        super.onDestroy()
        destroy(_mraidContentController)
        _adPlaylistController?.destroy()
        _adPlaylistController = null
        MraidActivityDataHolder.onAdClose?.invoke()

        scope.cancel()
    }

    // "Weak ref" stuff should work fine for single activity instance, since there's only one fullscreen ad displayed at time.
    // If that's not enough or in case of some simultaneous multi ad display edge cases:
    // implement fullscreen ad manager with storage like map: <UUID_or_int_increment, Pair<MraidController, Activity>>
    // where bundle start activity intent contains the UUID_or_int_increment so you can match all needed components of the ad,
    // therefore it's easy to create multiple instances of the fullscreen ad.
    companion object {
        private const val TAG = "MraidActivity"
        private val currentMraidEvents = MutableSharedFlow<AdControllerEvent>()

        /**
         * Displays the MRAID (Mobile Rich Media Ad Interface Definitions) ad in fullscreen mode.
         *
         * @param adData Data related to the MRAID ad, including ad content and metadata.
         * @param controller Controller responsible for managing the fullscreen ad's lifecycle and interactions.
         * @param context Context in which the ad should be displayed, typically the calling Activity or Application context.
         * @param options Configuration options for the ad WebView, such as JavaScript settings and ad view properties.
         * @param watermark Optional watermark to be displayed over the ad, for branding or copyright purposes.
         * @param onAdClose Optional callback function triggered when the ad is closed by the user. This is triggered upon [Activity.onDestroy] call
         */
        fun show(
                adData: MraidAdData,
                controller: MraidFullscreenContentController,
                context: Context,
                options: AdWebViewOptions,
                watermark: Watermark?,
                onAdClose: (() -> Unit)?,
        ): Boolean {
            if (!destroy(controller)) return false

            MraidActivityDataHolder.adData = adData
            MraidActivityDataHolder.watermark = watermark
            MraidActivityDataHolder.mraidRenderer = options.AdWebViewRenderer
            MraidActivityDataHolder.closeButton = options.decClose
            MraidActivityDataHolder.onAdClose = onAdClose
            MraidActivityDataHolder.weakController = controller

            context.startActivity(
                Intent(context, MraidActivity::class.java).apply {
                    putCloseDelaySeconds(options.closeDelaySeconds)
                    putDECDelaySeconds(options.decGoNextActionDelaySeconds)
                    flags = Intent.FLAG_ACTIVITY_NEW_TASK
                }
            )

            return true
        }

        private val AdControllerEvent.isErrorOrDismiss: Boolean
            get() = this is AdControllerEvent.Error || this == AdControllerEvent.Dismiss

        // TODO. Check if controller is the same to avoid destroying not owned activities.
        //  Ugly but works. Will change when have time/mood
        //  to something like: suspend fun show() with destroy logic during scope cancellation.
        private fun destroy(controller: MraidFullscreenContentController?): Boolean =
            with(MraidActivityDataHolder) {
                val wc = weakController
                // Treat null as correct value just in case :)
                return if (wc == null || wc == controller) {
                    weakController = null

                    // As ugly as it gets: remove webview to avoid attach/detach issues for banners.
                    (wc?.webView?.parent as? ViewGroup)?.removeView(wc.webView)

                    mraidRenderer = null
                    adData = null
                    watermark = null

                    weakActivity?.finish()
                    weakActivity = null
                    true
                } else {
                    false
                }
            }
    }
}

@VisibleForTesting
internal object MraidActivityDataHolder {

    private var _weakController = WeakReference<MraidFullscreenContentController?>(null)

    var weakController: MraidFullscreenContentController?
        get() = _weakController.get()
        set(value) {
            _weakController = WeakReference(value)
        }

    private var _weakActivity = WeakReference<Activity?>(null)

    var weakActivity: Activity?
        get() = _weakActivity.get()
        set(value) {
            _weakActivity = WeakReference(value)
        }

    var mraidRenderer: AdWebViewRenderer? = null
    var closeButton: @Composable () -> AdCountdownButton? = { defaultAdCloseCountdownButton() }
    var onAdClose: (() -> Unit)? = null
    var adData: MraidAdData? = null
    var watermark: Watermark? = null
}
