package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.content.Context
import android.graphics.Bitmap
import android.net.Uri
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import androidx.webkit.WebViewAssetLoader
import androidx.webkit.WebViewClientCompat
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow

// TODO. Duplication of VastWebViewClient, StaticWebViewClient and others.
internal class MraidWebViewClient(
    context: Context,
    private val mraidJsCommandsSource: MraidJsCommandUrlSource
) : WebViewClientCompat() {

    private val assetLoader = WebViewAssetLoader.Builder()
        .addPathHandler("/", WebViewAssetLoader.AssetsPathHandler(context.applicationContext))
        .build()

    override fun shouldInterceptRequest(
        view: WebView?,
        request: WebResourceRequest,
    ): WebResourceResponse? = assetLoader.shouldInterceptRequest(resolveMraidJsPath(request))

    private val _isLoaded = MutableStateFlow(false)
    val isLoaded: StateFlow<Boolean> = _isLoaded

    override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
        super.onPageStarted(view, url, favicon)
        _isLoaded.value = false
    }

    override fun onPageFinished(view: WebView?, url: String?) {
        super.onPageFinished(view, url)
        _isLoaded.value = true
    }

    private val _unrecoverableError = MutableStateFlow<MraidAdError?>(null)
    val unrecoverableError = _unrecoverableError.asStateFlow()

    /**
     * Report an error to the host application. These errors are unrecoverable (i.e. the main resource is unavailable). The errorCode parameter corresponds to one of the ERROR_* constants.
     */
    // Looking for unrecoverable errors only, hence deprecated function is better for that (?)
    // This annotation is required by Kotlin in order to avoid warnings.
    @Suppress("DEPRECATION")
    @Deprecated("Deprecated in Android API 23")
    override fun onReceivedError(
        view: WebView?,
        errorCode: Int,
        description: String?,
        failingUrl: String?,
    ) {
        super.onReceivedError(view, errorCode, description, failingUrl)
        _unrecoverableError.value = MraidAdError.MRAID_WEBVIEW_CLIENT_UNRECOVERABLE_ERROR
        MolocoLogger.error(TAG, "onReceivedError $description")
    }

    override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
        // TODO. Logging.
        // https://developer.android.com/guide/webapps/managing-webview#termination-handle
        // Basically, then webview will be destroyed externally after this, which, ideally, isn't known here.
        _unrecoverableError.value = MraidAdError.MRAID_WEBVIEW_CLIENT_RENDER_PROCESS_GONE_ERROR
        MolocoLogger.error(TAG, "onRenderProcessGone")
        return true
    }

    // Old version of the method works for api 33 as well, so I leave it as is.
    // This annotation is required by Kotlin in order to avoid warnings.
    @Deprecated("Deprecated in Android API 24")
    override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
        url ?: return false
        return mraidJsCommandsSource.consumeMraidJsCommand(url)
    }

    companion object {
        private const val TAG = "MraidWebViewClient"

        private const val ASSETS_MRAID_JS = "com.moloco.sdk.xenoss.sdkdevkit.mraid.js"
        private const val MRAID_JS_MATCHER = "mraid.js"

        // Replacing mraid.js with assets version with different name to avoid mraid.js conflicts,
        // when the app tries to include other sdks with the same asset filename.
        // Basic logic here.
        private fun resolveMraidJsPath(request: WebResourceRequest): Uri = Uri.parse(
            (request.url?.toString() ?: "").replace(
                MRAID_JS_MATCHER,
                ASSETS_MRAID_JS,
                ignoreCase = true
            )
        )
    }
}
