package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast

import com.moloco.sdk.internal.LogToBackend
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.utcNowEpochMillis
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VastError
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.ButtonLayoutSnapshot
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import java.net.URLEncoder
import java.util.concurrent.TimeUnit
import kotlin.random.Random

internal fun VastTracker(): VastTracker = Instance

private val Instance by lazy {
    VastTrackerImpl(
        SdkObjectFactory.Network.persistentHttpRequestSingleton
    )
}

interface VastTracker {
    /**
     * Tracks clicks with additional Moloco Telepathy events
     */
    fun trackClick(
        urls: List<String>,
        error: VastError? = null,
        contentPlayHead: Int? = null,
        assetUri: String? = null,
        renderedButtons: ButtonLayoutSnapshot,
        customUserEventBuilderService: CustomUserEventBuilderService,
        lastClickPosition: CustomUserEventBuilderService.UserInteraction.Position
    )

    fun track(
        urls: List<String>,
        error: VastError? = null,
        contentPlayHead: Int? = null,
        assetUri: String? = null
    )
}

internal class VastTrackerImpl(
    private val persistentHttpRequest: PersistentHttpRequest,
) : VastTracker {
    private val scope = CoroutineScope(DispatcherProvider().default)

    override fun trackClick(
        urls: List<String>,
        error: VastError?,
        contentPlayHead: Int?,
        assetUri: String?,
        renderedButtons: ButtonLayoutSnapshot,
        customUserEventBuilderService: CustomUserEventBuilderService,
        lastClickPosition: CustomUserEventBuilderService.UserInteraction.Position
    ) {
        track(
            urls,
            error,
            contentPlayHead,
            assetUri,
            renderedButtons = renderedButtons,
            customUserEventBuilderService = customUserEventBuilderService,
            lastClickPosition = lastClickPosition
        )
    }

    override fun track(
        urls: List<String>,
        error: VastError?,
        contentPlayHead: Int?,
        assetUri: String?
    ) {
        track(
            urls,
            error,
            contentPlayHead,
            assetUri,
            renderedButtons = listOf(),
            customUserEventBuilderService = null,
            lastClickPosition = null
        )
    }

    private fun track(
        urls: List<String>,
        error: VastError?,
        contentPlayHead: Int?,
        assetUri: String?,
        renderedButtons: ButtonLayoutSnapshot,
        customUserEventBuilderService: CustomUserEventBuilderService?,
        lastClickPosition: CustomUserEventBuilderService.UserInteraction.Position?,
    ) {
        if (urls.isEmpty()) {
            return
        }

        scope.launch {
            urls.forEach { url ->
                val urlWithQueryParams = if (customUserEventBuilderService != null && lastClickPosition != null) {
                    url.addUserInteractionQueryParams(
                        customUserEventBuilderService = customUserEventBuilderService,
                        lastClickPosition = lastClickPosition,
                        clickTimestamp = utcNowEpochMillis(),
                        renderedButtons = renderedButtons
                    )
                } else {
                    url
                }

                val urlWithQueryParamsAndSubstitutedMacros = urlWithQueryParams.substituteMacros(
                    errorCode = error?.value,
                    contentPlayHead = contentPlayHead,
                    assetUri = assetUri,
                    cacheBusting = createCacheBustingString()
                )

                persistentHttpRequest.send(urlWithQueryParamsAndSubstitutedMacros)
            }
        }
    }

    /**
     * Modifies the URL by adding user interaction query parameters.
     *
     * @param customUserEventBuilderService The service to build custom user event queries.
     * @param lastClickPosition The position of the last user interaction.
     * @param clickTimestamp The timestamp of the click event.
     * @param renderedButtons The layout snapshot of rendered buttons.
     * @return The modified URL with query parameters.
     */
    private suspend fun String.addUserInteractionQueryParams(
        customUserEventBuilderService: CustomUserEventBuilderService,
        lastClickPosition: CustomUserEventBuilderService.UserInteraction.Position,
        clickTimestamp: Long,
        renderedButtons: ButtonLayoutSnapshot,
    ) = customUserEventBuilderService.userAdInteractionExtAsQueryParameter(
        eventTimestamp = clickTimestamp,
        interaction = CustomUserEventBuilderService.UserInteraction.Click(
            clickPosition = lastClickPosition,
            buttonLayout = renderedButtons
        ),
        url = this
    )
}

private fun createCacheBustingString() = String.format("%08d", Random.nextInt(1, 99_999_999))

/**
 * Replaces macros in a VAST URL template with provided values.
 *
 * @param errorCode The error code to substitute into the URL, if present.
 * @param contentPlayHead The current position of the content playhead, in seconds, to substitute into the URL, if present.
 * @param assetUri The URI of the media asset to substitute into the URL, if present.
 * @param cacheBusting A cache-busting string to substitute into the URL, if present.
 * @return The URL with substituted macros.
 */
private fun String.substituteMacros(
    errorCode: Int?,
    contentPlayHead: Int?,
    assetUri: String?,
    cacheBusting: String?
): String {
    var res = this

    errorCode?.let { res = res.replace(ERRORCODE, errorCode.toString()) }
    contentPlayHead?.let {
        // Vast 3.0 - 4.1
        res = res.replace(CONTENTPLAYHEAD, contentPlayHead.formatContentPlayHead())

        // Vast 4.1 support
        res = res.replace(ADPLAYHEAD, contentPlayHead.formatAdPlayHead())
        res = res.replace(MEDIAPLAYHEAD, contentPlayHead.formatMediaPlayHead())
    }
    assetUri?.let { res = res.replace(ASSETURI, assetUri.urlEncode()) }
    cacheBusting?.let { res = res.replace(CACHEBUSTING, cacheBusting) }

    res = res.replace(ANYMACROS, "")

    return res
}

private fun Int.formatContentPlayHead(): String {
    val thisLong = toLong()

    return String.format(
        "%02d:%02d:%02d.%03d",
        TimeUnit.MILLISECONDS.toHours(thisLong),
        TimeUnit.MILLISECONDS.toMinutes(thisLong) % TimeUnit.HOURS.toMinutes(1),
        TimeUnit.MILLISECONDS.toSeconds(thisLong) % TimeUnit.MINUTES.toSeconds(1),
        thisLong % 1000
    )
}

private fun Int.formatAdPlayHead() = formatContentPlayHead()

// Not supported on the SDK client because
// ADPLAYHEAD is the time elapsed within the ad video and
// MEDIAPLAYHEAD is the time elapsed w.r.t the CONTENT in our
// case its the game which is non linear time
// Usually if there is youtube video with a length of say 10min
// and lets say if an ad plays at 5th min then initially
// ADPLAYHEAD = 0
// MEDIAPLAYHEAD = 5
private fun Int.formatMediaPlayHead() = "-1"

private fun String.urlEncode(): String {
    // URL-encode any URLs
    return try {
        URLEncoder.encode(this, "UTF-8")
    } catch (e: Exception) {
        @LogToBackend
        // TODO. Logging.
        ""
    }
}

// Vast 3.0
// https://www.iab.com/wp-content/uploads/2015/06/VASTv3_0.pdf
private val ERRORCODE = """\[ERRORCODE]""".toRegex()
private val CONTENTPLAYHEAD = """\[CONTENTPLAYHEAD]""".toRegex()
private val CACHEBUSTING = """\[CACHEBUSTING]""".toRegex()
private val ASSETURI = """\[ASSETURI]""".toRegex()
private val ANYMACROS = """\[[^]]*]""".toRegex()

// Vast 4.0
// https://interactiveadvertisingbureau.github.io/vast/vast4macros/vast4-macros-latest.html#macro-spec-mediaplayhead
private val MEDIAPLAYHEAD = """\[MEDIAPLAYHEAD]""".toRegex()
private val ADPLAYHEAD = """\[ADPLAYHEAD]""".toRegex()
