package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer

import android.view.View
import androidx.annotation.MainThread
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.flowOf
import kotlinx.coroutines.flow.launchIn

/**
 * Visibility aware video player:
 * automatically resumes/suspends playback if player view is invisible (off-screen, [View.isShown] == false etc)
 */
@MainThread
internal class VisibilityAwareVideoPlayer(
    private val basePlayer: SimplifiedExoPlayer,
    viewVisibilityTracker: ViewVisibilityTracker
) : VideoPlayer by basePlayer {

    private val scope = MainScope()

    // SharedFlow is better than StateFlow here since it'll trigger combine's action block { }
    // each time a new value is emitted (even equal to the old one),
    // also it doesn't trigger an unnecessary initial play()/pause() command in combine's action block { }.
    private val shouldPlay = MutableSharedFlow<Boolean>(
        replay = 1,
        onBufferOverflow = BufferOverflow.DROP_OLDEST
    )

    init {
        // Play only in case the player view is visible on the screen.
        combine(
            viewVisibilityTracker.isVisibleFlow(basePlayer.playerView),
            shouldPlay
        ) { isPlayerViewVisible, shouldPlay ->
            with(basePlayer) {
                if (isPlayerViewVisible && shouldPlay) {
                    play()
                } else {
                    pause()
                }
            }
        }.launchIn(scope)
    }

    override fun play() {
        shouldPlay.tryEmit(true)
    }

    override fun pause() {
        shouldPlay.tryEmit(false)
    }

    override fun destroy() {
        scope.cancel()
        basePlayer.destroy()
    }
}

// Nullable view support; return visibility "false" in this case.
private fun ViewVisibilityTracker.isVisibleFlow(view: View?) =
    view?.let { isVisibleFlow(it) } ?: flowOf(false)
