package com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.service_locator.SdkObjectFactory
import io.ktor.http.ContentType

/**
 * Best attempt persistent Http request that tries to initialize PersistentHttpRequest and if
 * fails so, then uses NonPersistentRequest. Used in cases such as crash reporting, where
 * we can no longer rely on the [WorkManager] still being functional.
 */
interface BestAttemptPersistentHttpRequest : HttpRequestClient

fun BestAttemptPersistentHttpRequest(): BestAttemptPersistentHttpRequest = Instance

@VisibleForTesting
internal fun provideInstance() = BestAttemptPersistentHttpRequestImpl(
    try {
        SdkObjectFactory.Network.persistentHttpRequestSingleton
    } catch (e: Exception) {
        MolocoLogger.warn(
            "BestAttemptHttpRequest",
            "Failed to create PersistentHttpRequest, " +
                "invoking NonPersistendHttpRequest",
            e
        )
        NonPersistentHttpRequest()
    }
)

private val Instance by lazy {
    provideInstance()
}

internal class BestAttemptPersistentHttpRequestImpl(private val delegate: HttpRequestClient) :
    BestAttemptPersistentHttpRequest {
    override fun send(url: String) {
        delegate.send(url)
    }

    override fun sendPost(url: String, body: ByteArray, contentType: ContentType) {
        delegate.sendPost(url, body, contentType)
    }
}
