package com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport

import android.content.Context
import androidx.annotation.VisibleForTesting
import androidx.work.Configuration
import androidx.work.Operation
import androidx.work.WorkManager
import androidx.work.WorkRequest
import com.moloco.sdk.internal.MolocoLogger
import java.util.UUID

interface PersistentWorker {
    fun enqueue(workRequest: WorkRequest): Operation
}

// TODO. IMPORTANT. Make sure WorkManager.getInstance(context)
//  doesn't interfere with publisher's WorkManager setup
//  https://developer.android.com/topic/libraries/architecture/workmanager/advanced/custom-configuration
//  https://developer.android.com/reference/androidx/work/DelegatingWorkerFactory
//  https://stackoverflow.com/questions/58343759/workmanager-doesnt-work-in-library-when-client-calls-workmanager-initialize
internal class PersistentWorkerImpl(context: Context) : PersistentWorker {
    private val _workManager: WorkManager

    init {
        var workManager: WorkManager
        try {
            workManager = tryGetWorkManagerInstance(context)
        } catch (e: IllegalStateException) {
            // See crash - https://mlc.atlassian.net/browse/SDK-969
            // If we get a WorkManager not initialized exception,
            // then try to initialize the WorkManager by ourselves and use that

            MolocoLogger.error(
                "MolocoWorkManager",
                "WorkManager not initialized already, performing initialization",
                e
            )
            workManager = tryInitializeWorkManagerManually(context)
        }
        _workManager = workManager
    }

    override fun enqueue(workRequest: WorkRequest): Operation = _workManager.enqueue(workRequest)

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    fun workInfoById(id: UUID) = _workManager.getWorkInfoById(id)

    private fun tryGetWorkManagerInstance(context: Context) = WorkManager.getInstance(context)

    /**
     * In cases where an exception is thrown when we initialize WorkManager by calling method [tryGetWorkManagerInstance] then
     * we try to initialize the WorkManager by ourselves and use that.
     * See crash - https://mlc.atlassian.net/browse/SDK-969
     */
    private fun tryInitializeWorkManagerManually(context: Context): WorkManager {
        val workManagerConfig = Configuration.Builder()
            .build()

        try {
            // Initialize can fail if someone else does initialization on another thread -
            // https://android.googlesource.com/platform/frameworks/support/+/60ae0eec2a32396c22ad92502cde952c80d514a0/work/workmanager/src/main/java/androidx/work/impl/WorkManagerImpl.java#177
            MolocoLogger.info(
                "MolocoWorkManager",
                "Trying to initialize work manager as one is not already available"
            )
            WorkManager.initialize(context, workManagerConfig)
        } catch (e: IllegalStateException) {
            // This is getting pretty hacky to work around the bug- https://issuetracker.google.com/issues/258176803
            // However theoretically it is possible that by the time we tried to initialize,
            // the startup provider on another thread tried to initialize WorkManager
            // (It should only happen on main thread, but this check is fallback safety mechanism)
            // In that case we don't crash and gracefully handle it
            MolocoLogger.error(
                "MolocoWorkManager",
                "WorkManager initialized already at this point, retrieving instance",
                e
            )
        }

        // Try to retrieve the WM instance again one last time as either we should have initialized
        // or it should have been initialized by someone else, but an instance should have been
        // created by this point
        MolocoLogger.info("MolocoWorkManager", "Trying to retrieve work manager instance")
        try {
            return tryGetWorkManagerInstance(context)
        } catch (e: IllegalStateException) {
            MolocoLogger.warn(
                "MolocoWorkManager",
                "WorkManager instance couldn't be re-initialized, cannot provide WorkManager"
            )
            throw IllegalStateException(
                "Cannot provide MolocoWorkManager. Failed to re-initialize WorkManager",
                e
            )
        }
    }
}
