package com.moloco.sdk.internal

import com.moloco.sdk.internal.ortb.BidResponseParser
import com.moloco.sdk.internal.publisher.BidPreprocessor
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.publisher.MolocoAdError.ErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType

/**
 * Moloco Internal Ad error object that hides the sub error type from the public API. This is used internally
 * for logging / analytics / debugging purposes.
 */
internal data class MolocoInternalAdError(
    val molocoAdError: MolocoAdError, // Public API visible error
    val subErrorType: MolocoAdSubErrorType, // Internal error type (Not visible to public)
)

/**
 * Some errors in [ErrorType] will have more granular subtypes. This enum represents those subtypes. These are
 * errors happening at the SDK layer (not at the renderer layer). Render errors are handled separately in the xenoss package
 */
enum class InternalSDKErrorSubType : MolocoAdSubErrorType {
    AD_LOAD_LIMIT_REACHED, // Ad cap limit reached

    /**
     * Bid could not be processed by the [BidPreprocessor]
     */
    BID_LOAD_ERROR_CANNOT_PROCESS_BID_RESPONSE,

    /**
     * Bid could not be parsed by the [BidResponseParser]
     */
    BID_LOAD_ERROR_CANNOT_PARSE_BID_RESPONSE,


    AD_SHOW_ERROR_NOT_LOADED,
    AD_SHOW_ERROR_ALREADY_DISPLAYING,
    // Add more SDK subtypes here as needed (not xenoss, which are handled in xenoss package)
    ;

    override val metricsRepresentation: String
        get() = name
}

/**
 * Creates a new [MolocoInternalAdError] object with [MolocoAdError.networkName] set to "Moloco".
 *
 * @param adUnitId The id of the ad unit.
 * @param errorType The type of the error.
 * @param subErrorType The sub type of the error. This is not visible to the public API.
 * @return A new [MolocoAdError] object with [MolocoAdError.networkName] set to "Moloco".
 */
internal fun createInternalAdErrorInfo(
    adUnitId: String,
    errorType: ErrorType,
    subErrorType: MolocoAdSubErrorType,
): MolocoInternalAdError = MolocoInternalAdError(MolocoAdError("Moloco", adUnitId, errorType), subErrorType)