package com.moloco.sdk.internal.scheduling

import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlin.coroutines.CoroutineContext

/**
 * Interface representing a provider of coroutine dispatchers.
 */
interface DispatcherProvider {
    /**
     * The main coroutine dispatcher, typically associated with the main UI thread.
     */
    val main: CoroutineContext

    /**
     * The mainImmediate coroutine dispatcher, which starts coroutines immediately on the main thread
     * if it's already in the current call stack, avoiding an extra dispatch.
     */
    val mainImmediate: CoroutineContext

    /**
     * The default coroutine dispatcher, typically used for CPU-intensive tasks.
     */
    val default: CoroutineContext

    /**
     * The IO coroutine dispatcher, typically used for IO-bound tasks such as network operations or file IO.
     */
    val io: CoroutineContext

    /**
     * The unconfined coroutine dispatcher, which executes tasks without any specific confinement.
     */
    val unconfined: CoroutineContext
}

/**
 * Returns an instance of [DispatcherProvider] implementation.
 */
fun DispatcherProvider(): DispatcherProvider = DispatcherProviderImpl()

private class DispatcherProviderImpl : DispatcherProvider {
    override val main: CoroutineDispatcher
        get() = Dispatchers.Main

    override val mainImmediate: CoroutineDispatcher
        get() = Dispatchers.Main.immediate

    override val default: CoroutineDispatcher
        get() = Dispatchers.Default
    override val io: CoroutineDispatcher
        get() = Dispatchers.IO
    override val unconfined: CoroutineDispatcher
        get() = Dispatchers.Unconfined
}
